// System Detection and Compatibility Checker
const os = require('os');
const { exec } = require('child_process');
const fs = require('fs');
const path = require('path');

class SystemDetector {
  constructor() {
    this.systemInfo = {};
  }

  async detectSystem() {
    console.log('🔍 Detecting system specifications...');
    
    try {
      // Basic system info
      this.systemInfo.platform = os.platform();
      this.systemInfo.arch = os.arch();
      this.systemInfo.totalMemory = Math.round(os.totalmem() / (1024 * 1024 * 1024));
      this.systemInfo.cpus = os.cpus().length;
      this.systemInfo.hostname = os.hostname();
      
      // Windows version detection
      if (this.systemInfo.platform === 'win32') {
        await this.detectWindowsVersion();
        await this.detectInstalledSoftware();
        await this.detectPrinters();
      }
      
      return this.systemInfo;
    } catch (error) {
      console.error('Error detecting system:', error);
      return this.systemInfo;
    }
  }

  async detectWindowsVersion() {
    return new Promise((resolve) => {
      exec('ver', (error, stdout) => {
        if (!error) {
          this.systemInfo.windowsVersion = stdout.trim();
          
          // Determine Windows compatibility
          if (stdout.includes('10.0')) {
            this.systemInfo.windowsCompatible = true;
            this.systemInfo.windowsMajor = '10/11';
          } else if (stdout.includes('6.3')) {
            this.systemInfo.windowsCompatible = true;
            this.systemInfo.windowsMajor = '8.1';
          } else if (stdout.includes('6.1')) {
            this.systemInfo.windowsCompatible = false;
            this.systemInfo.windowsMajor = '7';
          } else {
            this.systemInfo.windowsCompatible = false;
            this.systemInfo.windowsMajor = 'Unknown';
          }
        }
        resolve();
      });
    });
  }

  async detectInstalledSoftware() {
    return new Promise((resolve) => {
      exec('powershell "Get-WmiObject -Class Win32_Product | Select-Object Name | ConvertTo-Json"', (error, stdout) => {
        if (!error) {
          try {
            const software = JSON.parse(stdout);
            this.systemInfo.installedSoftware = Array.isArray(software) ? software.length : 0;
          } catch (e) {
            this.systemInfo.installedSoftware = 0;
          }
        }
        resolve();
      });
    });
  }

  async detectPrinters() {
    return new Promise((resolve) => {
      exec('powershell "Get-WmiObject -Class Win32_Printer | Select-Object Name, Manufacturer | ConvertTo-Json"', (error, stdout) => {
        if (!error) {
          try {
            const printers = JSON.parse(stdout);
            this.systemInfo.printers = Array.isArray(printers) ? printers : [printers];
            this.systemInfo.printerCount = this.systemInfo.printers.length;
          } catch (e) {
            this.systemInfo.printers = [];
            this.systemInfo.printerCount = 0;
          }
        }
        resolve();
      });
    });
  }

  generateCompatibilityReport() {
    const report = {
      compatible: true,
      warnings: [],
      recommendations: [],
      systemSpecs: this.systemInfo
    };

    // Check Windows compatibility
    if (!this.systemInfo.windowsCompatible) {
      report.compatible = false;
      report.warnings.push(`Windows ${this.systemInfo.windowsMajor} is not supported. Requires Windows 10 or later.`);
    }

    // Check memory
    if (this.systemInfo.totalMemory < 4) {
      report.warnings.push(`Low memory detected (${this.systemInfo.totalMemory}GB). 4GB+ recommended.`);
    }

    // Check architecture
    if (this.systemInfo.arch !== 'x64') {
      report.warnings.push(`32-bit system detected. 64-bit Windows recommended for best performance.`);
    }

    // Printer recommendations
    if (this.systemInfo.printerCount === 0) {
      report.recommendations.push('No printers detected. Connect printers for full functionality.');
    } else {
      report.recommendations.push(`${this.systemInfo.printerCount} printer(s) detected. Ready for driver management.`);
    }

    return report;
  }

  async createAdaptiveConfig() {
    const config = {
      system: this.systemInfo,
      features: {
        driverUpdates: this.systemInfo.windowsCompatible,
        printerManagement: this.systemInfo.printerCount > 0,
        securityScanning: true,
        userAuthentication: true
      },
      performance: {
        animationsEnabled: this.systemInfo.totalMemory >= 8,
        backgroundScanning: this.systemInfo.cpus >= 4,
        cacheSize: Math.min(this.systemInfo.totalMemory * 100, 1000)
      },
      ui: {
        theme: 'professional',
        compactMode: this.systemInfo.totalMemory < 8,
        highContrast: false
      }
    };

    // Save config for application use
    const configPath = path.join(__dirname, 'adaptive-config.json');
    fs.writeFileSync(configPath, JSON.stringify(config, null, 2));
    
    return config;
  }
}

module.exports = SystemDetector;

// CLI usage
if (require.main === module) {
  const detector = new SystemDetector();
  
  detector.detectSystem().then(() => {
    console.log('\n📊 System Detection Complete:');
    console.log('================================');
    console.log(`Platform: ${detector.systemInfo.platform}`);
    console.log(`Architecture: ${detector.systemInfo.arch}`);
    console.log(`Memory: ${detector.systemInfo.totalMemory}GB`);
    console.log(`CPUs: ${detector.systemInfo.cpus}`);
    console.log(`Windows: ${detector.systemInfo.windowsMajor}`);
    console.log(`Printers: ${detector.systemInfo.printerCount}`);
    
    const report = detector.generateCompatibilityReport();
    console.log(`\n✅ Compatible: ${report.compatible}`);
    
    if (report.warnings.length > 0) {
      console.log('\n⚠️  Warnings:');
      report.warnings.forEach(warning => console.log(`  - ${warning}`));
    }
    
    if (report.recommendations.length > 0) {
      console.log('\n💡 Recommendations:');
      report.recommendations.forEach(rec => console.log(`  - ${rec}`));
    }
    
    detector.createAdaptiveConfig().then(() => {
      console.log('\n🔧 Adaptive configuration created: adaptive-config.json');
    });
  });
}
