const { app, BrowserWindow, ipcMain, dialog, Notification, shell } = require('electron');
const path = require('path');
const { exec } = require('child_process');
const fs = require('fs');
const crypto = require('crypto');
const os = require('os');
const https = require('https');
const { spawn } = require('child_process');

let mainWindow;
let scanningInterval;
let updateCheckInterval;
let securityScanInterval;
let notificationQueue = [];
let isScanning = false;
let lastScanTime = 0;

// License management
const LICENSE_FILE = path.join(os.homedir(), 'AppData', 'Local', 'GreekDriverUpdater', 'license.json');
const INSTALL_FILE = path.join(os.homedir(), 'AppData', 'Local', 'GreekDriverUpdater', 'install.json');
const USERS_FILE = path.join(os.homedir(), 'AppData', 'Local', 'GreekDriverUpdater', 'users.json');
const SESSION_FILE = path.join(os.homedir(), 'AppData', 'Local', 'GreekDriverUpdater', 'session.json'); // removed duplicate SESSION_FILE
const SECURITY_CONFIG_FILE = path.join(os.homedir(), 'AppData', 'Local', 'GreekDriverUpdater', 'security.json');
const ACTIVATION_AUDIT_FILE = path.join(os.homedir(), 'AppData', 'Local', 'GreekDriverUpdater', 'activation_audit.json');
const TRIAL_DAYS = 7;
const SECRET_KEY = 'greeksquad-driver-updater-2024';

// Generate license key
function generateLicenseKey(email, expiryDate) {
  const data = `${email}:${expiryDate}`;
  const hash = crypto.createHmac('sha256', SECRET_KEY).update(data).digest('hex');
  return `GDU-${hash.substring(0, 8).toUpperCase()}-${hash.substring(8, 16).toUpperCase()}-${hash.substring(16, 24).toUpperCase()}`;
}

// -------- Activation hardening helpers --------
function getDeviceFingerprint() {
  try {
    const nets = os.networkInterfaces();
    const macs = Object.values(nets).flat().filter(Boolean).map(n => n.mac).filter(m => m && m !== '00:00:00:00:00:00');
    const data = [os.hostname(), os.platform(), os.arch(), ...macs.sort()].join('|');
    return crypto.createHash('sha256').update(data).digest('hex');
  } catch {
    return crypto.createHash('sha256').update(os.hostname()).digest('hex');
  }
}

function recordActivationAttempt(entry) {
  try {
    const dir = path.dirname(ACTIVATION_AUDIT_FILE);
    if (!fs.existsSync(dir)) fs.mkdirSync(dir, { recursive: true });
    const existing = fs.existsSync(ACTIVATION_AUDIT_FILE) ? JSON.parse(fs.readFileSync(ACTIVATION_AUDIT_FILE, 'utf8')) : [];
    existing.push({ timestamp: Date.now(), ...entry });
    fs.writeFileSync(ACTIVATION_AUDIT_FILE, JSON.stringify(existing.slice(-1000), null, 2));
  } catch {}
}

function tooManyRecentActivations(limit = 5, windowMs = 60 * 60 * 1000) {
  try {
    if (!fs.existsSync(ACTIVATION_AUDIT_FILE)) return false;
    const entries = JSON.parse(fs.readFileSync(ACTIVATION_AUDIT_FILE, 'utf8')) || [];
    const since = Date.now() - windowMs;
    const recent = entries.filter(e => e && e.timestamp >= since);
    return recent.length >= limit;
  } catch {
    return false;
  }
}

async function checkWindowsFirewallEnabled() {
  return new Promise((resolve) => {
    exec('powershell "(Get-NetFirewallProfile).Enabled | ConvertTo-Json"', (err, stdout) => {
      if (err) return resolve({ enabled: null });
      try {
        const res = JSON.parse(stdout);
        const arr = Array.isArray(res) ? res : [res];
        // If any profile enabled, consider firewall enabled
        resolve({ enabled: arr.some(v => String(v).toLowerCase() === 'true') });
      } catch {
        resolve({ enabled: null });
      }
    });
  });
}

async function isAdminUser() {
  return new Promise((resolve) => {
    exec('whoami /groups', (err, stdout) => {
      if (err) return resolve(false);
      resolve(/S-1-5-32-544/i.test(stdout)); // Administrators SID
    });
  });
}

// Validate license key
function validateLicenseKey(licenseKey, email) {
  try {
    const parts = licenseKey.split('-');
    if (parts.length !== 4 || parts[0] !== 'GDU') return false;
    
    const providedHash = parts.slice(1).join('').toLowerCase();
    
    // Try different expiry dates to find a match
    const currentDate = new Date();
    for (let years = 1; years <= 10; years++) {
      const testExpiry = new Date(currentDate.getFullYear() + years, 11, 31).getTime();
      const testData = `${email}:${testExpiry}`;
      const testHash = crypto.createHmac('sha256', SECRET_KEY).update(testData).digest('hex');
      
      if (testHash.substring(0, 24) === providedHash) {
        return { valid: true, expiryDate: testExpiry };
      }
    }
    
    return false;
  } catch (error) {
    return false;
  }
}

// Real-time notification system
function showNotification(title, body, type = 'info') {
  if (Notification.isSupported()) {
    const notification = new Notification({
      title: `Greek Driver Updater - ${title}`,
      body: body,
      icon: path.join(__dirname, '../public/assets/greek-squad-logo.jpg'),
      urgency: type === 'critical' ? 'critical' : 'normal',
      timeoutType: type === 'critical' ? 'never' : 'default'
    });

    notification.on('click', () => {
      if (mainWindow) {
        mainWindow.show();
        mainWindow.focus();
      }
    });

    notification.show();
    
    // Log notification
    writeLog(`Notification: ${title} - ${body}`);
  }
}

// Enhanced driver downloading with real-time updates
async function downloadDriverDirectly(manufacturer, model) {
  return new Promise((resolve, reject) => {
    const driverUrls = {
      'HP': `https://support.hp.com/drivers/selfservice/swdetails/hp-${model.toLowerCase().replace(/\s+/g, '-')}`,
      'Canon': `https://www.canon-europe.com/support/consumer_products/products/printers/${model.toLowerCase().replace(/\s+/g, '_')}.html`,
      'Epson': `https://epson.com/Support/Printers/${model.replace(/\s+/g, '-')}/s/SPT_${model.replace(/\s+/g, '')}`,
      'Brother': `https://support.brother.com/g/b/downloadtop.aspx?c=us&lang=en&prod=${model.replace(/\s+/g, '')}`,
      'Samsung': `https://www.samsung.com/us/support/mobile-devices/${model.toLowerCase().replace(/\s+/g, '-')}/`,
      'Lexmark': `https://www.lexmark.com/en_us/support/drivers-downloads.html?model=${model}`,
      'Xerox': `https://www.support.xerox.com/support/product-support/${model.toLowerCase().replace(/\s+/g, '-')}`,
      'Ricoh': `https://www.ricoh-usa.com/en/support-and-download/drivers/${model.toLowerCase().replace(/\s+/g, '-')}`
    };

    const url = driverUrls[manufacturer];
    if (!url) {
      reject(new Error(`Driver not available for ${manufacturer}`));
      return;
    }

    // Show download notification
    showNotification('Driver Download', `Starting download for ${manufacturer} ${model}`, 'info');
    
    // Open manufacturer's driver page
    shell.openExternal(url);
    
    // Log the download attempt
    writeLog(`Driver download initiated: ${manufacturer} ${model} - URL: ${url}`);
    
    resolve({
      success: true,
      manufacturer: manufacturer,
      model: model,
      url: url,
      message: `Driver download page opened for ${manufacturer} ${model}`
    });
  });
}

// Real-time system scanning
async function performSystemScan() {
  if (isScanning) return;
  
  isScanning = true;
  lastScanTime = Date.now();
  
  try {
    showNotification('System Scan', 'Starting comprehensive system scan...', 'info');
    
    const scanResults = {
      driversFound: 0,
      updatesAvailable: 0,
      securityThreats: 0,
      systemHealth: 'good',
      recommendations: []
    };

    // Scan for outdated drivers
    const driverScan = await scanForOutdatedDrivers();
    scanResults.driversFound = driverScan.length;
    
    if (driverScan.length > 0) {
      scanResults.updatesAvailable += driverScan.length;
      scanResults.recommendations.push(`${driverScan.length} driver updates available`);
      
      // Show critical notification for driver updates
      showNotification('Driver Updates Available', 
        `Found ${driverScan.length} driver updates for your system`, 'critical');
    }

    // Scan for Windows updates
    const windowsUpdates = await checkWindowsUpdates();
    if (windowsUpdates.available > 0) {
      scanResults.updatesAvailable += windowsUpdates.available;
      scanResults.recommendations.push(`${windowsUpdates.available} Windows updates available`);
      
      showNotification('Windows Updates', 
        `${windowsUpdates.available} Windows updates are ready to install`, 'critical');
    }

    // Security scan
    const securityScan = await performSecurityScan();
    scanResults.securityThreats = securityScan.threats;
    
    if (securityScan.threats > 0) {
      scanResults.systemHealth = 'warning';
      scanResults.recommendations.push(`${securityScan.threats} security issues detected`);
      
      showNotification('Security Alert', 
        `Detected ${securityScan.threats} potential security threats`, 'critical');
    }

    // System performance check
    const performanceCheck = await checkSystemPerformance();
    if (performanceCheck.score < 70) {
      scanResults.systemHealth = 'poor';
      scanResults.recommendations.push('System performance optimization recommended');
      
      showNotification('Performance Warning', 
        'System performance is below optimal levels', 'info');
    }

    writeLog(`System scan completed: ${JSON.stringify(scanResults)}`);
    
    return scanResults;
    
  } catch (error) {
    writeLog(`System scan error: ${error.message}`);
    showNotification('Scan Error', 'System scan encountered an error', 'info');
    return null;
  } finally {
    isScanning = false;
  }
}

// Enhanced driver scanning
async function scanForOutdatedDrivers() {
  return new Promise((resolve) => {
    exec('powershell "Get-WmiObject Win32_PnPSignedDriver | Where-Object {$_.DeviceName -like \'*printer*\' -or $_.DeviceName -like \'*scanner*\'} | Select-Object DeviceName, DriverVersion, DriverDate | ConvertTo-Json"', 
      (error, stdout) => {
        if (error) {
          resolve([]);
          return;
        }
        
        try {
          const drivers = JSON.parse(stdout);
          const outdatedDrivers = [];
          
          const driversArray = Array.isArray(drivers) ? drivers : [drivers];
          
          driversArray.forEach(driver => {
            if (driver.DriverDate) {
              const driverDate = new Date(driver.DriverDate);
              const monthsOld = (Date.now() - driverDate.getTime()) / (1000 * 60 * 60 * 24 * 30);
              
              if (monthsOld > 6) { // Consider drivers older than 6 months as outdated
                outdatedDrivers.push({
                  name: driver.DeviceName,
                  version: driver.DriverVersion,
                  date: driver.DriverDate,
                  monthsOld: Math.floor(monthsOld)
                });
              }
            }
          });
          
          resolve(outdatedDrivers);
        } catch (e) {
          resolve([]);
        }
      }
    );
  });
}

// Windows update checking
async function checkWindowsUpdates() {
  return new Promise((resolve) => {
    exec('powershell "Get-WindowsUpdate -AcceptAll -Download:$false | Measure-Object | Select-Object Count"', 
      (error, stdout) => {
        if (error) {
          resolve({ available: 0, error: error.message });
          return;
        }
        
        try {
          const result = stdout.trim();
          const count = parseInt(result.match(/\d+/)?.[0] || '0');
          resolve({ available: count });
        } catch (e) {
          resolve({ available: 0 });
        }
      }
    );
  });
}

// Enhanced security scanning
async function performSecurityScan() {
  return new Promise((resolve) => {
    let threats = 0;
    const securityChecks = [];

    // Check for suspicious processes
    exec('powershell "Get-Process | Where-Object {$_.ProcessName -match \'malware|virus|trojan|keylog\'} | Measure-Object | Select-Object Count"', 
      (error, stdout) => {
        if (!error) {
          const suspiciousProcesses = parseInt(stdout.match(/\d+/)?.[0] || '0');
          threats += suspiciousProcesses;
        }
        
        // Check for unusual network connections
        exec('powershell "Get-NetTCPConnection | Where-Object {$_.State -eq \'Established\' -and $_.RemotePort -in @(6667,6697,1337,31337)} | Measure-Object | Select-Object Count"', 
          (error2, stdout2) => {
            if (!error2) {
              const suspiciousConnections = parseInt(stdout2.match(/\d+/)?.[0] || '0');
              threats += suspiciousConnections;
            }
            
            // Check Windows Defender status
            exec('powershell "Get-MpComputerStatus | Select-Object AntivirusEnabled, RealTimeProtectionEnabled"', 
              (error3, stdout3) => {
                let defenderIssues = 0;
                if (!error3 && stdout3.includes('False')) {
                  defenderIssues = 1;
                  threats += defenderIssues;
                }
                
                resolve({ 
                  threats: threats,
                  details: {
                    suspiciousProcesses: threats > 0,
                    networkThreats: threats > 0,
                    defenderDisabled: defenderIssues > 0
                  }
                });
              }
            );
          }
        );
      }
    );
  });
}

// System performance check
async function checkSystemPerformance() {
  return new Promise((resolve) => {
    exec('powershell "Get-Counter \'\\Processor(_Total)\\% Processor Time\' -SampleInterval 1 -MaxSamples 3 | Select-Object -ExpandProperty CounterSamples | Measure-Object CookedValue -Average"', 
      (error, stdout) => {
        let cpuScore = 100;
        if (!error) {
          const avgCpu = parseFloat(stdout.match(/[\d.]+/)?.[0] || '0');
          cpuScore = Math.max(0, 100 - avgCpu);
        }
        
        exec('powershell "Get-Counter \'\\Memory\\Available MBytes\' | Select-Object -ExpandProperty CounterSamples | Select-Object CookedValue"', 
          (error2, stdout2) => {
            let memoryScore = 100;
            if (!error2) {
              const availableMemory = parseInt(stdout2.match(/\d+/)?.[0] || '0');
              memoryScore = Math.min(100, (availableMemory / 1024) * 25); // Score based on available GB
            }
            
            const overallScore = (cpuScore + memoryScore) / 2;
            resolve({ 
              score: Math.round(overallScore),
              cpu: Math.round(cpuScore),
              memory: Math.round(memoryScore)
            });
          }
        );
      }
    );
  });
}

// Get license status
function getLicenseStatus() {
  try {
    if (!fs.existsSync(LICENSE_FILE)) {
      // No license file, check if trial period
      const installDate = getInstallDate();
      const daysSinceInstall = Math.floor((Date.now() - installDate) / (1000 * 60 * 60 * 24));
      
      if (daysSinceInstall < TRIAL_DAYS) {
        return {
          status: 'trial',
          daysRemaining: TRIAL_DAYS - daysSinceInstall,
          message: `Trial period: ${TRIAL_DAYS - daysSinceInstall} days remaining`
        };
      } else {
        return {
          status: 'expired',
          message: 'Trial period expired. Please activate your license.'
        };
      }
    }
    
    const licenseData = JSON.parse(fs.readFileSync(LICENSE_FILE, 'utf8'));
    const validation = validateLicenseKey(licenseData.key, licenseData.email);
    
    if (!validation) {
      return {
        status: 'invalid',
        message: 'Invalid license key'
      };
    }
    
    if (Date.now() > validation.expiryDate) {
      return {
        status: 'expired',
        message: 'License has expired'
      };
    }
    // Enforce device binding if stored
    try {
      if (licenseData.fingerprint) {
        const currentFp = getDeviceFingerprint();
        if (licenseData.fingerprint !== currentFp) {
          return { status: 'invalid', message: 'License is bound to another device' };
        }
      }
    } catch {}
    
    return {
      status: 'active',
      email: licenseData.email,
      expiryDate: validation.expiryDate,
      message: 'License is active'
    };
  } catch (error) {
    return {
      status: 'error',
      message: 'Error checking license status'
    };
  }
}

// Get install date
function getInstallDate() {
  const installFile = INSTALL_FILE;
  if (fs.existsSync(installFile)) {
    try {
      const data = JSON.parse(fs.readFileSync(installFile, 'utf8'));
      if (data && data.installDate) return data.installDate;
    } catch (error) {
      // fall through to recreate
    }
  }
  const installDate = Date.now();
  const dir = path.dirname(installFile);
  if (!fs.existsSync(dir)) fs.mkdirSync(dir, { recursive: true });
  fs.writeFileSync(installFile, JSON.stringify({ installDate }));
  return installDate;
}

// User management functions
function hashPassword(password) {
  return crypto.createHash('sha256').update(password + SECRET_KEY).digest('hex');
}

function loadUsers() {
  if (!fs.existsSync(USERS_FILE)) {
    return {};
  }
  try {
    return JSON.parse(fs.readFileSync(USERS_FILE, 'utf8'));
  } catch (error) {
    return {};
  }
}

function saveUsers(users) {
  const dir = path.dirname(USERS_FILE);
  if (!fs.existsSync(dir)) {
    fs.mkdirSync(dir, { recursive: true });
  }
  fs.writeFileSync(USERS_FILE, JSON.stringify(users, null, 2));
}

function generateSessionToken() {
  return crypto.randomBytes(32).toString('hex');
}

function saveSession(sessionData) {
  fs.writeFileSync(SESSION_FILE, JSON.stringify(sessionData, null, 2));
}

function loadSession() {
  if (!fs.existsSync(SESSION_FILE)) {
    return null;
  }
  try {
    const session = JSON.parse(fs.readFileSync(SESSION_FILE, 'utf8'));
    // Check if session is expired (24 hours)
    if (Date.now() - session.timestamp > 24 * 60 * 60 * 1000) {
      fs.unlinkSync(SESSION_FILE);
      return null;
    }
    return session;
  } catch (error) {
    return null;
  }
}

function clearSession() {
  if (fs.existsSync(SESSION_FILE)) {
    fs.unlinkSync(SESSION_FILE);
  }
}

function createWindow() {
  mainWindow = new BrowserWindow({
    width: 1200,
    height: 800,
    webPreferences: {
      nodeIntegration: false,
      contextIsolation: true,
      preload: path.join(__dirname, 'preload.js')
    },
    icon: path.join(__dirname, 'assets', 'icon.png'),
    show: false
  });

  const isDev = process.env.NODE_ENV === 'development';
  
  if (isDev) {
    mainWindow.loadURL('http://localhost:3000');
    mainWindow.webContents.openDevTools();
  } else {
    mainWindow.loadFile(path.join(__dirname, '../build/index.html'));
  }

  mainWindow.once('ready-to-show', () => {
    mainWindow.show();
  });

  mainWindow.on('closed', () => {
    mainWindow = null;
  });
}

app.whenReady().then(createWindow);

app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') {
    app.quit();
  }
});

app.on('activate', () => {
  if (BrowserWindow.getAllWindows().length === 0) {
    createWindow();
  }
});

// User authentication IPC handlers
ipcMain.handle('register-user', async (event, userData) => {
  try {
    const { email, password, firstName, lastName, company } = userData;
    
    if (!email || !password || !firstName || !lastName) {
      return { success: false, message: 'All fields are required' };
    }
    
    const users = loadUsers();
    
    if (users[email]) {
      return { success: false, message: 'User already exists' };
    }
    
    const hashedPassword = hashPassword(password);
    users[email] = {
      email,
      password: hashedPassword,
      firstName,
      lastName,
      company: company || '',
      createdAt: Date.now(),
      lastLogin: null,
      profile: {
        avatar: null,
        phone: '',
        address: '',
        preferences: {
          notifications: true,
          autoUpdates: true
        }
      }
    };
    
    saveUsers(users);
    
    return { success: true, message: 'User registered successfully' };
  } catch (error) {
    return { success: false, message: 'Registration failed' };
  }
});

ipcMain.handle('login-user', async (event, credentials) => {
  try {
    const { email, password } = credentials;
    
    if (!email || !password) {
      return { success: false, message: 'Email and password are required' };
    }
    
    const users = loadUsers();
    const user = users[email];
    
    if (!user) {
      return { success: false, message: 'User not found' };
    }
    
    const hashedPassword = hashPassword(password);
    if (user.password !== hashedPassword) {
      return { success: false, message: 'Invalid password' };
    }
    
    // Update last login
    user.lastLogin = Date.now();
    saveUsers(users);
    
    // Create session
    const sessionToken = generateSessionToken();
    const sessionData = {
      token: sessionToken,
      email: user.email,
      timestamp: Date.now()
    };
    
    saveSession(sessionData);
    
    return { 
      success: true, 
      message: 'Login successful',
      user: {
        email: user.email,
        firstName: user.firstName,
        lastName: user.lastName,
        company: user.company,
        profile: user.profile
      },
      token: sessionToken
    };
  } catch (error) {
    return { success: false, message: 'Login failed' };
  }
});

ipcMain.handle('logout-user', async () => {
  try {
    clearSession();
    return { success: true, message: 'Logged out successfully' };
  } catch (error) {
    return { success: false, message: 'Logout failed' };
  }
});

ipcMain.handle('get-current-user', async () => {
  try {
    const session = loadSession();
    if (!session) {
      return { success: false, message: 'No active session' };
    }
    
    const users = loadUsers();
    const user = users[session.email];
    
    if (!user) {
      clearSession();
      return { success: false, message: 'User not found' };
    }
    
    return {
      success: true,
      user: {
        email: user.email,
        firstName: user.firstName,
        lastName: user.lastName,
        company: user.company,
        profile: user.profile,
        createdAt: user.createdAt,
        lastLogin: user.lastLogin
      }
    };
  } catch (error) {
    return { success: false, message: 'Failed to get user data' };
  }
});

ipcMain.handle('update-user-profile', async (event, profileData) => {
  try {
    const session = loadSession();
    if (!session) {
      return { success: false, message: 'No active session' };
    }
    
    const users = loadUsers();
    const user = users[session.email];
    
    if (!user) {
      return { success: false, message: 'User not found' };
    }
    
    // Update user profile
    user.firstName = profileData.firstName || user.firstName;
    user.lastName = profileData.lastName || user.lastName;
    user.company = profileData.company || user.company;
    
    if (profileData.profile) {
      user.profile = { ...user.profile, ...profileData.profile };
    }
    
    saveUsers(users);
    
    return { success: true, message: 'Profile updated successfully' };
  } catch (error) {
    return { success: false, message: 'Profile update failed' };
  }
});

ipcMain.handle('change-password', async (event, passwordData) => {
  try {
    const { currentPassword, newPassword } = passwordData;
    const session = loadSession();
    
    if (!session) {
      return { success: false, message: 'No active session' };
    }
    
    const users = loadUsers();
    const user = users[session.email];
    
    if (!user) {
      return { success: false, message: 'User not found' };
    }
    
    const hashedCurrentPassword = hashPassword(currentPassword);
    if (user.password !== hashedCurrentPassword) {
      return { success: false, message: 'Current password is incorrect' };
    }
    
    user.password = hashPassword(newPassword);
    saveUsers(users);
    
    return { success: true, message: 'Password changed successfully' };
  } catch (error) {
    return { success: false, message: 'Password change failed' };
  }
});

// License IPC handlers
ipcMain.handle('get-license-status', async () => {
  return getLicenseStatus();
});

ipcMain.handle('activate-license', async (event, licenseKey, email) => {
  try {
    // Rate limit
    if (tooManyRecentActivations()) {
      writeSecurityLog('Activation rate-limited', { email });
      return { success: false, message: 'Too many activation attempts. Please wait and try again.' };
    }
    const validation = validateLicenseKey(licenseKey, email);
    
    if (!validation) {
      recordActivationAttempt({ email, ok: false, reason: 'invalid-key' });
      return { success: false, message: 'Invalid license key' };
    }
    
    if (Date.now() > validation.expiryDate) {
      recordActivationAttempt({ email, ok: false, reason: 'expired' });
      return { success: false, message: 'License key has expired' };
    }
    
    // Save license
    const fingerprint = getDeviceFingerprint();
    const licenseData = {
      key: licenseKey,
      email: email,
      activatedDate: Date.now(),
      fingerprint
    };
    
    fs.writeFileSync(LICENSE_FILE, JSON.stringify(licenseData, null, 2));
    // Activation audit: environment checks
    const admin = await isAdminUser();
    const defender = await performSecurityScan();
    const firewall = await checkWindowsFirewallEnabled();
    const audit = {
      email,
      admin,
      defenderIssues: defender?.details?.defenderDisabled || false,
      firewallEnabled: firewall.enabled,
      timestamp: new Date().toISOString(),
    };
    writeSecurityLog('Activation audit', audit);
    recordActivationAttempt({ email, ok: true, ...audit });
    
    return { 
      success: true, 
      message: 'License activated successfully',
      expiryDate: validation.expiryDate,
      audit
    };
  } catch (error) {
    recordActivationAttempt({ email, ok: false, reason: 'exception', error: error.message });
    return { success: false, message: 'Error activating license' };
  }
});

ipcMain.handle('deactivate-license', async () => {
  try {
    if (fs.existsSync(LICENSE_FILE)) {
      fs.unlinkSync(LICENSE_FILE);
    }
    return { success: true, message: 'License deactivated' };
  } catch (error) {
    return { success: false, message: 'Error deactivating license' };
  }
});

ipcMain.handle('generate-license-key', async (event, email, years = 1) => {
  try {
    const expiryDate = new Date(new Date().getFullYear() + years, 11, 31).getTime();
    const licenseKey = generateLicenseKey(email, expiryDate);
    return { 
      success: true, 
      licenseKey,
      expiryDate,
      message: 'License key generated successfully'
    };
  } catch (error) {
    return { success: false, message: 'Error generating license key' };
  }
});

// Driver download URLs by manufacturer - Security validated URLs
const DRIVER_URLS = {
  'HP': 'https://support.hp.com/drivers',
  'Canon': 'https://www.canon-europe.com/support/products/',
  'Epson': 'https://epson.com/support/drivers-and-downloads',
  'Brother': 'https://support.brother.com/g/b/downloadtop.aspx',
  'Samsung': 'https://www.samsung.com/us/support/downloads/',
  'Lexmark': 'https://support.lexmark.com/en_us/drivers-downloads.html',
  'Xerox': 'https://www.support.xerox.com/support/product-support/drivers-downloads',
  'Ricoh': 'https://www.ricoh-usa.com/en/support-and-download/drivers'
};

// Security blacklist for malicious websites
const SECURITY_BLACKLIST = [
  'suspicious-driver-site.com',
  'fake-hp-drivers.net',
  'unknown-printer-tools.org',
  'malware-drivers.xyz',
  'phishing-canon.com',
  'virus-epson.net',
  'trojan-brother.org'
];

// Security functions
function isUrlSafe(url) {
  try {
    const domain = new URL(url).hostname.toLowerCase();
    const cfg = loadSecurityConfig();
    const dynamicBlacklist = (cfg.blacklist || []);
    const whitelist = (cfg.whitelist || []);
    if (whitelist.some(w => domain.endsWith(w.toLowerCase()))) return true;
    return ![...SECURITY_BLACKLIST, ...dynamicBlacklist].some(blocked => domain.includes(blocked.toLowerCase()));
  } catch (error) {
    return false;
  }
}

function logSecurityEvent(eventType, url, severity = 'medium') {
  const timestamp = new Date().toISOString();
  const logEntry = `[SECURITY] ${timestamp} - ${eventType}: ${url} (Severity: ${severity})`;
  console.log(logEntry);
  
  // Write to security log file
  const securityLogPath = path.join(os.homedir(), 'AppData', 'Local', 'GreekDriverUpdater', 'security.log');
  try {
    fs.appendFileSync(securityLogPath, logEntry + '\n');
  } catch (error) {
    console.error('Failed to write security log:', error);
  }
}

// IPC handlers for driver operations with security validation
ipcMain.handle('check-driver-updates', async () => {
  return new Promise((resolve, reject) => {
    const command = `powershell -Command "Get-WmiObject -Class Win32_Printer | Select-Object Name, DriverName, PortName, Manufacturer | ConvertTo-Json"`;
    
    exec(command, (error, stdout, stderr) => {
      if (error) {
        logSecurityEvent('SYSTEM_SCAN_ERROR', 'check-driver-updates', 'low');
        reject({ error: error.message, stderr });
        return;
      }
      
      try {
        const printers = JSON.parse(stdout);
        const printerArray = Array.isArray(printers) ? printers : [printers];
        
        const brandKeys = Object.keys(DRIVER_URLS); // Canonical brand names
        const driversToUpdate = printerArray.map(printer => {
          const mLower = (printer.Manufacturer || '').toLowerCase();
          const brand = brandKeys.find(b => mLower.includes(b.toLowerCase())) || 'Unknown';
          return {
            name: printer.Name,
            driver: printer.DriverName,
            manufacturer: printer.Manufacturer,
            brand: brand,
            downloadUrl: DRIVER_URLS[brand] || null,
            hasUpdate: brand !== 'Unknown'
          };
        });
        
        resolve({ drivers: driversToUpdate });
      } catch (parseError) {
        reject({ error: 'Failed to parse printer data', parseError: parseError.message });
      }
    });
  });
});

// Keep legacy handler name to avoid channel collision; canonical handler defined below
ipcMain.handle('download-driver-legacy', async (event, driverInfo) => {
  return new Promise((resolve, reject) => {
    if (!driverInfo || !driverInfo.downloadUrl) {
      reject({ error: 'No download URL available for this manufacturer' });
      return;
    }
    require('electron').shell.openExternal(driverInfo.downloadUrl);
    resolve({ success: true, message: `Opened ${driverInfo.manufacturer} driver download page`, url: driverInfo.downloadUrl });
  });
});

// Enhanced real-time monitoring and notifications
function startRealTimeMonitoring() {
  // Start automatic system scanning every 30 minutes
  scanningInterval = setInterval(async () => {
    const scanResult = await performSystemScan();
    if (scanResult && mainWindow) {
      mainWindow.webContents.send('scan-completed', scanResult);
    }
  }, 30 * 60 * 1000); // 30 minutes

  // Check for Windows updates every 2 hours
  updateCheckInterval = setInterval(async () => {
    const updates = await checkWindowsUpdates();
    if (updates.available > 0) {
      showNotification('Windows Updates Available', 
        `${updates.available} Windows updates are ready to install`, 'critical');
      
      if (mainWindow) {
        mainWindow.webContents.send('updates-available', updates);
      }
    }
  }, 2 * 60 * 60 * 1000); // 2 hours

  // Security monitoring every 15 minutes
  securityScanInterval = setInterval(async () => {
    const securityResult = await performSecurityScan();
    if (securityResult.threats > 0) {
      showNotification('Security Alert', 
        `Detected ${securityResult.threats} potential security threats`, 'critical');
      
      if (mainWindow) {
        mainWindow.webContents.send('security-alert', securityResult);
      }
    }
  }, 15 * 60 * 1000); // 15 minutes

  // Initial scan on startup
  setTimeout(async () => {
    showNotification('System Monitoring Active', 
      'Greek Driver Updater is now monitoring your system for updates and threats', 'info');
    
    const initialScan = await performSystemScan();
    if (initialScan && mainWindow) {
      mainWindow.webContents.send('initial-scan-complete', initialScan);
    }
  }, 5000); // 5 seconds after startup
}

function stopRealTimeMonitoring() {
  if (scanningInterval) clearInterval(scanningInterval);
  if (updateCheckInterval) clearInterval(updateCheckInterval);
  if (securityScanInterval) clearInterval(securityScanInterval);
}

// Enhanced app update detection
async function checkForAppUpdates() {
  return new Promise((resolve) => {
    // Check for common applications that need updates
    const appsToCheck = [
      'Google Chrome',
      'Mozilla Firefox',
      'Adobe Acrobat',
      'Java',
      'VLC Media Player',
      'Zoom',
      'Microsoft Teams',
      'Skype',
      'Discord',
      'Steam'
    ];

    exec('powershell "Get-WmiObject -Class Win32_Product | Select-Object Name, Version | ConvertTo-Json"', 
      (error, stdout) => {
        if (error) {
          resolve({ available: 0, apps: [] });
          return;
        }

        try {
          const installedApps = JSON.parse(stdout);
          const appsArray = Array.isArray(installedApps) ? installedApps : [installedApps];
          
          const outdatedApps = [];
          
          appsArray.forEach(app => {
            if (appsToCheck.some(checkApp => app.Name && app.Name.includes(checkApp))) {
              // Simulate version checking (in real implementation, you'd check against latest versions)
              const isOutdated = Math.random() > 0.7; // 30% chance of being outdated for demo
              if (isOutdated) {
                outdatedApps.push({
                  name: app.Name,
                  currentVersion: app.Version,
                  needsUpdate: true
                });
              }
            }
          });

          resolve({ 
            available: outdatedApps.length, 
            apps: outdatedApps 
          });
        } catch (e) {
          resolve({ available: 0, apps: [] });
        }
      }
    );
  });
}

// Enhanced security event logging
function writeSecurityLog(event, details) {
  const logEntry = {
    timestamp: new Date().toISOString(),
    event: event,
    details: details,
    severity: 'security',
    sessionId: Date.now().toString()
  };
  
  const logDir = path.join(os.homedir(), 'AppData', 'Local', 'GreekDriverUpdater');
  if (!fs.existsSync(logDir)) {
    fs.mkdirSync(logDir, { recursive: true });
  }
  
  const logFile = path.join(logDir, 'security.log');
  fs.appendFileSync(logFile, JSON.stringify(logEntry) + '\n');
  
  // Also send to main window if available
  if (mainWindow) {
    mainWindow.webContents.send('security-log-entry', logEntry);
  }
}

function writeLog(message) {
  const logDir = path.join(os.homedir(), 'AppData', 'Local', 'GreekDriverUpdater');
  if (!fs.existsSync(logDir)) {
    fs.mkdirSync(logDir, { recursive: true });
  }
  
  const logFile = path.join(logDir, 'app.log');
  const timestamp = new Date().toISOString();
  fs.appendFileSync(logFile, `[${timestamp}] ${message}\n`);
}

// Expose writeLog to renderer
ipcMain.handle('write-log', async (event, message) => {
  try {
    writeLog(String(message || ''));
    return { success: true };
  } catch (e) {
    return { success: false, message: e.message };
  }
});

// Enhanced IPC handlers with real-time features
ipcMain.handle('download-driver', async (event, manufacturer, model) => {
  try {
    // Security check for manufacturer and model
    if (!manufacturer || !model) {
      writeSecurityLog('Invalid driver request', { manufacturer, model });
      return { success: false, message: 'Invalid manufacturer or model' };
    }
    
    const result = await downloadDriverDirectly(manufacturer, model);
    
    // Log successful driver download
    writeSecurityLog('Driver download', { 
      manufacturer, 
      model, 
      url: result.url,
      success: true 
    });
    
    return result;
  } catch (error) {
    writeSecurityLog('Driver download error', { 
      manufacturer, 
      model, 
      error: error.message 
    });
    
    return { success: false, message: error.message };
  }
});

ipcMain.handle('scan-system', async () => {
  try {
    const scanResult = await performSystemScan();
    
    writeSecurityLog('System scan completed', {
      driversFound: scanResult?.driversFound || 0,
      securityThreats: scanResult?.securityThreats || 0,
      systemHealth: scanResult?.systemHealth || 'unknown'
    });
    
    return { success: true, data: scanResult };
  } catch (error) {
    writeSecurityLog('System scan error', { error: error.message });
    return { success: false, message: 'System scan failed' };
  }
});

ipcMain.handle('check-app-updates', async () => {
  try {
    const updates = await checkForAppUpdates();
    
    if (updates.available > 0) {
      showNotification('Application Updates Available', 
        `${updates.available} applications have updates available`, 'info');
    }
    
    writeSecurityLog('App update check', {
      updatesAvailable: updates.available,
      apps: updates.apps.map(app => app.name)
    });
    
    return { success: true, data: updates };
  } catch (error) {
    writeSecurityLog('App update check error', { error: error.message });
    return { success: false, message: 'Failed to check for app updates' };
  }
});

ipcMain.handle('start-monitoring', async () => {
  try {
    startRealTimeMonitoring();
    writeSecurityLog('Real-time monitoring started', { timestamp: Date.now() });
    return { success: true, message: 'Real-time monitoring started' };
  } catch (error) {
    return { success: false, message: 'Failed to start monitoring' };
  }
});

ipcMain.handle('stop-monitoring', async () => {
  try {
    stopRealTimeMonitoring();
    writeSecurityLog('Real-time monitoring stopped', { timestamp: Date.now() });
    return { success: true, message: 'Real-time monitoring stopped' };
  } catch (error) {
    return { success: false, message: 'Failed to stop monitoring' };
  }
});

ipcMain.handle('get-system-status', async () => {
  try {
    const performance = await checkSystemPerformance();
    const security = await performSecurityScan();
    const drivers = await scanForOutdatedDrivers();
    const windowsUpdates = await checkWindowsUpdates();
    const appUpdates = await checkForAppUpdates();
    
    const status = {
      performance: performance,
      security: security,
      drivers: drivers,
      windowsUpdates: windowsUpdates,
      appUpdates: appUpdates,
      isMonitoring: !!scanningInterval,
      lastScan: lastScanTime
    };
    
    return { success: true, data: status };
  } catch (error) {
    return { success: false, message: 'Failed to get system status' };
  }
});

// Printer management functions
ipcMain.handle('get-printers', async () => {
  return new Promise((resolve) => {
    exec('powershell "Get-WmiObject -Class Win32_Printer | Select-Object Name, Status, Default | ConvertTo-Json"', (error, stdout) => {
      if (error) {
        resolve([]);
        return;
      }
      
      try {
        const printers = JSON.parse(stdout);
        resolve(Array.isArray(printers) ? printers : [printers]);
      } catch (e) {
        resolve([]);
      }
    });
  });
});

ipcMain.handle('get-printer-drivers', async () => {
  return new Promise((resolve) => {
    exec('powershell "Get-WmiObject -Class Win32_PrinterDriver | Select-Object Name, Version | ConvertTo-Json"', (error, stdout) => {
      if (error) {
        resolve([]);
        return;
      }
      
      try {
        const drivers = JSON.parse(stdout);
        resolve(Array.isArray(drivers) ? drivers : [drivers]);
      } catch (e) {
        resolve([]);
      }
    });
  });
});

ipcMain.handle('update-driver', async (event, manufacturer) => {
  return new Promise((resolve) => {
    const url = DRIVER_URLS[manufacturer];
    if (!url) {
      resolve({ success: false, message: 'Driver not available for this manufacturer' });
      return;
    }
    
    // Security check
    if (!isUrlSafe(url)) {
      writeSecurityLog('Blocked unsafe driver URL', { manufacturer, url });
      resolve({ success: false, message: 'Security: Unsafe driver source blocked' });
      return;
    }
    
    // Open manufacturer's driver page instead of direct download
    require('electron').shell.openExternal(url);
    writeSecurityLog('Driver update page opened', { manufacturer, url });
    
    resolve({
      success: true,
      message: `Driver update page opened for ${manufacturer}`,
      url: url
    });
  });
});

ipcMain.handle('read-logs', async () => {
  const logFile = path.join(os.homedir(), 'AppData', 'Local', 'GreekDriverUpdater', 'app.log');
  
  return new Promise((resolve, reject) => {
    fs.readFile(logFile, 'utf8', (error, data) => {
      if (error) {
        if (error.code === 'ENOENT') {
          resolve({ logs: [] });
          return;
        }
        reject(error);
        return;
      }
      
      const logs = data.split('\n').filter(line => line.trim()).reverse();
      resolve({ logs });
    });
  });
});

// ---------------- Safe Search and Firewall utilities ----------------
function loadSecurityConfig() {
  try {
    if (!fs.existsSync(SECURITY_CONFIG_FILE)) return { blacklist: [], whitelist: [], safeSearch: true };
    const data = JSON.parse(fs.readFileSync(SECURITY_CONFIG_FILE, 'utf8'));
    return { blacklist: data.blacklist || [], whitelist: data.whitelist || [], safeSearch: data.safeSearch !== false };
  } catch {
    return { blacklist: [], whitelist: [], safeSearch: true };
  }
}

function saveSecurityConfig(cfg) {
  const dir = path.dirname(SECURITY_CONFIG_FILE);
  if (!fs.existsSync(dir)) fs.mkdirSync(dir, { recursive: true });
  fs.writeFileSync(SECURITY_CONFIG_FILE, JSON.stringify(cfg, null, 2));
}

ipcMain.handle('security-get-config', async () => {
  return loadSecurityConfig();
});

ipcMain.handle('security-update-config', async (event, patch) => {
  const current = loadSecurityConfig();
  const next = {
    blacklist: Array.from(new Set([...(current.blacklist || []), ...((patch && patch.blacklist) || [])])),
    whitelist: Array.from(new Set([...(current.whitelist || []), ...((patch && patch.whitelist) || [])])),
    safeSearch: typeof patch?.safeSearch === 'boolean' ? patch.safeSearch : current.safeSearch
  };
  saveSecurityConfig(next);
  writeSecurityLog('Security config updated', next);
  return { success: true, config: next };
});

ipcMain.handle('security-validate-url', async (event, url) => {
  const safe = isUrlSafe(url || '');
  return { safe };
});

// Firewall management using PowerShell (requires admin)
function runPowershell(cmd) {
  return new Promise((resolve, reject) => {
    exec(`powershell -Command "${cmd}"`, (error, stdout, stderr) => {
      if (error) return reject(new Error(stderr || error.message));
      resolve(stdout);
    });
  });
}

ipcMain.handle('firewall-block-domain', async (event, domain) => {
  try {
    const lic = getLicenseStatus();
    if (!lic || lic.status !== 'active') {
      return { success: false, message: 'Firewall controls require an active license.' };
    }
    if (!domain) return { success: false, message: 'Domain required' };
    const ruleName = `GDU Block ${domain}`;
    // Use New-NetFirewallRule with RemoteFQDN (Windows 10/11)
    await runPowershell(`New-NetFirewallRule -DisplayName '${ruleName}' -Direction Outbound -Action Block -RemoteFqdn '${domain}' -Enabled True -Description 'Blocked by Greek Driver Updater'`);
    writeSecurityLog('Firewall block domain', { domain, ruleName });
    return { success: true };
  } catch (e) {
    writeSecurityLog('Firewall block failed', { domain, error: e.message });
    return { success: false, message: e.message };
  }
});

ipcMain.handle('firewall-unblock-domain', async (event, domain) => {
  try {
    const lic = getLicenseStatus();
    if (!lic || lic.status !== 'active') {
      return { success: false, message: 'Firewall controls require an active license.' };
    }
    if (!domain) return { success: false, message: 'Domain required' };
    const ruleName = `GDU Block ${domain}`;
    await runPowershell(`Get-NetFirewallRule -DisplayName '${ruleName}' | Remove-NetFirewallRule`);
    writeSecurityLog('Firewall unblock domain', { domain, ruleName });
    return { success: true };
  } catch (e) {
    writeSecurityLog('Firewall unblock failed', { domain, error: e.message });
    return { success: false, message: e.message };
  }
});

ipcMain.handle('firewall-list-rules', async () => {
  try {
    const lic = getLicenseStatus();
    if (!lic || lic.status !== 'active') {
      return { success: false, rules: [], message: 'Firewall controls require an active license.' };
    }
    const out = await runPowershell(`Get-NetFirewallRule -DisplayName 'GDU Block *' | Select-Object DisplayName, Enabled | ConvertTo-Json`);
    const json = out ? JSON.parse(out) : [];
    return { success: true, rules: Array.isArray(json) ? json : [json] };
  } catch (e) {
    return { success: false, rules: [], message: e.message };
  }
});

// ---------------- Application Updates via winget ----------------
ipcMain.handle('check-updates', async () => {
  return new Promise((resolve) => {
    // List available upgrades in JSON
    const cmd = 'winget upgrade --accept-source-agreements --include-unknown --output json';
    exec(cmd, { windowsHide: true }, (error, stdout) => {
      if (error) {
        writeSecurityLog('Winget check failed', { error: error.message });
        resolve({ updates: [], error: error.message });
        return;
      }
      try {
        const data = JSON.parse(stdout);
        const arr = Array.isArray(data) ? data : (data?.Sources || data?.Installed || []);
        const updates = (arr || [])
          .map(item => {
            // winget json includes fields: Id, Name, Version, Available, Source
            const id = item.Id || item.IdLower || item.PackageIdentifier || item.PackageId;
            const name = item.Name || item.PackageName || id;
            const currentVersion = item.Version || item.InstalledVersion || '';
            const availableVersion = item.Available || item.AvailableVersion || '';
            if (!availableVersion || availableVersion === currentVersion) return null;
            return { id, name, currentVersion, availableVersion };
          })
          .filter(Boolean);
        resolve({ updates });
      } catch (e) {
        resolve({ updates: [] });
      }
    });
  });
});

ipcMain.handle('upgrade-app', async (event, appId) => {
  return new Promise((resolve) => {
    const baseFlags = '--accept-package-agreements --accept-source-agreements --silent';
    const cmd = appId && typeof appId === 'string' && appId.trim().length > 0
      ? `winget upgrade --id "${appId}" ${baseFlags}`
      : `winget upgrade --all ${baseFlags}`;
    exec(cmd, { windowsHide: true }, (error, stdout, stderr) => {
      if (error) {
        writeSecurityLog('Winget upgrade failed', { appId: appId || 'all', error: error.message, stderr });
        resolve({ success: false, message: stderr || error.message });
        return;
      }
      writeSecurityLog('Winget upgrade success', { appId: appId || 'all' });
      resolve({ success: true });
    });
  });
});
