import React, { useState, useEffect } from 'react';
import { 
  CogIcon, 
  ClockIcon, 
  BellIcon, 
  ShieldCheckIcon,
  ComputerDesktopIcon,
  CheckCircleIcon
} from '@heroicons/react/24/outline';

const Settings = ({ licenseStatus, onLicenseUpdate }) => {
  const [settings, setSettings] = useState({
    autoUpdateCheck: true,
    updateFrequency: 'daily',
    silentUpdates: false,
    notifications: true,
    autoRestartSpooler: false,
    logLevel: 'info',
    startWithWindows: false
  });
  
  const [saved, setSaved] = useState(false);

  useEffect(() => {
    loadSettings();
  }, []);

  const loadSettings = () => {
    // Load settings from localStorage
    const savedSettings = localStorage.getItem('pcPrinterAssistantSettings');
    if (savedSettings) {
      setSettings(JSON.parse(savedSettings));
    }
  };

  const saveSettings = async () => {
    try {
      localStorage.setItem('pcPrinterAssistantSettings', JSON.stringify(settings));
      await window.electronAPI.writeLog(`Settings updated: ${JSON.stringify(settings)}`);
      setSaved(true);
      setTimeout(() => setSaved(false), 2000);
    } catch (error) {
      console.error('Failed to save settings:', error);
    }
  };

  const updateSetting = (key, value) => {
    setSettings(prev => ({
      ...prev,
      [key]: value
    }));
  };

  const SettingSection = ({ title, description, icon: Icon, children }) => (
    <div className="bg-white rounded-lg shadow mb-6">
      <div className="p-6 border-b border-gray-200">
        <div className="flex items-center">
          <Icon className="h-6 w-6 text-gray-400 mr-3" />
          <div>
            <h3 className="text-lg font-semibold text-gray-900">{title}</h3>
            <p className="text-sm text-gray-600 mt-1">{description}</p>
          </div>
        </div>
      </div>
      <div className="p-6 space-y-4">
        {children}
      </div>
    </div>
  );

  const ToggleSetting = ({ label, description, checked, onChange }) => (
    <div className="flex items-center justify-between">
      <div className="flex-1">
        <h4 className="text-sm font-medium text-gray-900">{label}</h4>
        <p className="text-sm text-gray-500">{description}</p>
      </div>
      <button
        type="button"
        className={`${
          checked ? 'bg-primary-600' : 'bg-gray-200'
        } relative inline-flex h-6 w-11 flex-shrink-0 cursor-pointer rounded-full border-2 border-transparent transition-colors duration-200 ease-in-out focus:outline-none focus:ring-2 focus:ring-primary-500 focus:ring-offset-2`}
        onClick={() => onChange(!checked)}
      >
        <span
          className={`${
            checked ? 'translate-x-5' : 'translate-x-0'
          } pointer-events-none inline-block h-5 w-5 transform rounded-full bg-white shadow ring-0 transition duration-200 ease-in-out`}
        />
      </button>
    </div>
  );

  const SelectSetting = ({ label, description, value, options, onChange }) => (
    <div className="flex items-center justify-between">
      <div className="flex-1">
        <h4 className="text-sm font-medium text-gray-900">{label}</h4>
        <p className="text-sm text-gray-500">{description}</p>
      </div>
      <select
        value={value}
        onChange={(e) => onChange(e.target.value)}
        className="mt-1 block w-32 rounded-md border-gray-300 py-2 pl-3 pr-10 text-base focus:border-primary-500 focus:outline-none focus:ring-primary-500 sm:text-sm"
      >
        {options.map((option) => (
          <option key={option.value} value={option.value}>
            {option.label}
          </option>
        ))}
      </select>
    </div>
  );

  return (
    <div className="p-6">
      <div className="mb-8">
        <h1 className="text-2xl font-bold text-gray-900">Settings</h1>
        <p className="text-gray-600 mt-1">Configure your preferences and automation settings</p>
      </div>

      <SettingSection
        title="Update Management"
        description="Configure how the application handles updates"
        icon={ShieldCheckIcon}
      >
        <ToggleSetting
          label="Automatic Update Checks"
          description="Automatically check for application updates"
          checked={settings.autoUpdateCheck}
          onChange={(value) => updateSetting('autoUpdateCheck', value)}
        />
        
        <SelectSetting
          label="Update Check Frequency"
          description="How often to check for updates"
          value={settings.updateFrequency}
          options={[
            { value: 'hourly', label: 'Hourly' },
            { value: 'daily', label: 'Daily' },
            { value: 'weekly', label: 'Weekly' },
            { value: 'monthly', label: 'Monthly' }
          ]}
          onChange={(value) => updateSetting('updateFrequency', value)}
        />
        
        <ToggleSetting
          label="Silent Updates"
          description="Install updates automatically without prompting"
          checked={settings.silentUpdates}
          onChange={(value) => updateSetting('silentUpdates', value)}
        />
      </SettingSection>

      <SettingSection
        title="Notifications"
        description="Control when and how you receive notifications"
        icon={BellIcon}
      >
        <ToggleSetting
          label="Enable Notifications"
          description="Show desktop notifications for important events"
          checked={settings.notifications}
          onChange={(value) => updateSetting('notifications', value)}
        />
      </SettingSection>

      <SettingSection
        title="Printer Management"
        description="Automated printer maintenance settings"
        icon={ComputerDesktopIcon}
      >
        <ToggleSetting
          label="Auto-restart Print Spooler"
          description="Automatically restart the print spooler when printer issues are detected"
          checked={settings.autoRestartSpooler}
          onChange={(value) => updateSetting('autoRestartSpooler', value)}
        />
      </SettingSection>

      <SettingSection
        title="System Integration"
        description="Integration with Windows system features"
        icon={CogIcon}
      >
        <ToggleSetting
          label="Start with Windows"
          description="Launch PC Printer Assistant when Windows starts"
          checked={settings.startWithWindows}
          onChange={(value) => updateSetting('startWithWindows', value)}
        />
        
        <SelectSetting
          label="Log Level"
          description="Amount of detail to include in logs"
          value={settings.logLevel}
          options={[
            { value: 'error', label: 'Errors Only' },
            { value: 'warn', label: 'Warnings' },
            { value: 'info', label: 'Information' },
            { value: 'debug', label: 'Debug' }
          ]}
          onChange={(value) => updateSetting('logLevel', value)}
        />
      </SettingSection>

      {/* Save Button */}
      <div className="flex justify-end space-x-3">
        <button
          onClick={saveSettings}
          className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-primary-600 hover:bg-primary-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-500"
        >
          {saved ? (
            <>
              <CheckCircleIcon className="h-4 w-4 mr-2" />
              Saved!
            </>
          ) : (
            'Save Settings'
          )}
        </button>
      </div>

      {/* Current Settings Summary */}
      <div className="bg-gray-50 rounded-lg p-6 mt-6">
        <h3 className="text-lg font-medium text-gray-900 mb-4">Current Configuration</h3>
        <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
          <div>
            <span className="font-medium text-gray-700">Auto Updates:</span>
            <span className="ml-2 text-gray-600">
              {settings.autoUpdateCheck ? `Enabled (${settings.updateFrequency})` : 'Disabled'}
            </span>
          </div>
          <div>
            <span className="font-medium text-gray-700">Silent Updates:</span>
            <span className="ml-2 text-gray-600">
              {settings.silentUpdates ? 'Enabled' : 'Disabled'}
            </span>
          </div>
          <div>
            <span className="font-medium text-gray-700">Notifications:</span>
            <span className="ml-2 text-gray-600">
              {settings.notifications ? 'Enabled' : 'Disabled'}
            </span>
          </div>
          <div>
            <span className="font-medium text-gray-700">Auto Spooler Restart:</span>
            <span className="ml-2 text-gray-600">
              {settings.autoRestartSpooler ? 'Enabled' : 'Disabled'}
            </span>
          </div>
          <div>
            <span className="font-medium text-gray-700">Start with Windows:</span>
            <span className="ml-2 text-gray-600">
              {settings.startWithWindows ? 'Enabled' : 'Disabled'}
            </span>
          </div>
          <div>
            <span className="font-medium text-gray-700">Log Level:</span>
            <span className="ml-2 text-gray-600 capitalize">{settings.logLevel}</span>
          </div>
        </div>
      </div>
    </div>
  );
};

export default Settings;
