  const [licenseStatus, setLicenseStatus] = useState(null);
  const safeSectionRef = React.useRef(null);
  const firewallSectionRef = React.useRef(null);
  const recentBlocksRef = React.useRef(null);
import React, { useState, useEffect } from 'react';
import { 
  ShieldCheckIcon,
  ExclamationTriangleIcon,
  GlobeAltIcon,
  EyeSlashIcon,
  CheckCircleIcon,
  XCircleIcon,
  ClockIcon,
  Cog6ToothIcon,
  ChartBarIcon,
  BoltIcon,
  PhoneIcon,
  EnvelopeIcon
} from '@heroicons/react/24/outline';

const SecurityCenter = () => {
  const [securityStats, setSecurityStats] = useState({
    threatsBlocked: 0,
    suspiciousWebsites: 0,
    safeConnections: 0,
    lastScan: null
  });
  
  const [securitySettings, setSecuritySettings] = useState({
    realTimeProtection: true,
    websiteBlocking: true,
    malwareProtection: true,
    phishingProtection: true,
    adultContentFilter: false,
    downloadScanning: true
  });
  
  const [recentBlocks, setRecentBlocks] = useState([]);
  const [scanning, setScanning] = useState(false);
  const [safeSearchEnabled, setSafeSearchEnabled] = useState(true);
  const [blacklist, setBlacklist] = useState([]);
  const [whitelist, setWhitelist] = useState([]);
  const [domainInput, setDomainInput] = useState('');
  const [rules, setRules] = useState([]);
  const [validation, setValidation] = useState(null);

  useEffect(() => {
    loadSecurityData();
    loadSecurityConfig();
    loadLicenseStatus();
    // Simulate real-time updates
    const interval = setInterval(() => {
      updateSecurityStats();
    }, 30000);
    
    return () => clearInterval(interval);
  }, []);

  const loadSecurityData = async () => {
    try {
      // Simulate loading security data
      setSecurityStats({
        threatsBlocked: Math.floor(Math.random() * 50) + 10,
        suspiciousWebsites: Math.floor(Math.random() * 25) + 5,
        safeConnections: Math.floor(Math.random() * 500) + 100,
        lastScan: new Date().toLocaleString()
      });
      
      setRecentBlocks([
        {
          id: 1,
          type: 'malware',
          url: 'suspicious-driver-site.com',
          timestamp: new Date(Date.now() - 1000 * 60 * 15).toLocaleString(),
          severity: 'high'
        },
        {
          id: 2,
          type: 'phishing',
          url: 'fake-hp-drivers.net',
          timestamp: new Date(Date.now() - 1000 * 60 * 45).toLocaleString(),
          severity: 'critical'
        },
        {
          id: 3,
          type: 'suspicious',
          url: 'unknown-printer-tools.org',
          timestamp: new Date(Date.now() - 1000 * 60 * 120).toLocaleString(),
          severity: 'medium'
        }
      ]);
    } catch (error) {
      console.error('Failed to load security data:', error);
    }
  };

  const loadSecurityConfig = async () => {
    try {
      const cfg = await window.electronAPI.securityGetConfig();
      setSafeSearchEnabled(!!cfg.safeSearch);
      setBlacklist(cfg.blacklist || []);
      setWhitelist(cfg.whitelist || []);
    } catch (e) {
      console.error('Failed to load security config', e);
    }
  };

  const toggleSafeSearch = async () => {
    try {
      const next = !safeSearchEnabled;
      await window.electronAPI.securityUpdateConfig({ safeSearch: next });
      setSafeSearchEnabled(next);
      await window.electronAPI.writeLog(`Safe Search set to ${next}`);
    } catch (e) {
      console.error('Failed to update safe search', e);
    }
  };

  const addToBlacklist = async () => {
    const domain = domainInput.trim();
    if (!domain) return;
    try {
      await window.electronAPI.securityUpdateConfig({ blacklist: [domain] });
      setBlacklist(prev => Array.from(new Set([...prev, domain])));
      setDomainInput('');
    } catch (e) {
      console.error('Failed to update blacklist', e);
    }
  };

  const addToWhitelist = async () => {
    const domain = domainInput.trim();
    if (!domain) return;
    try {
      await window.electronAPI.securityUpdateConfig({ whitelist: [domain] });
      setWhitelist(prev => Array.from(new Set([...prev, domain])));
      setDomainInput('');
    } catch (e) {
      console.error('Failed to update whitelist', e);
    }
  };

  const validateDomain = async () => {
    const domain = domainInput.trim();
    if (!domain) return;
    try {
      const { safe } = await window.electronAPI.securityValidateUrl(`http://${domain}`);
      setValidation(safe ? 'safe' : 'blocked');
    } catch (e) {
      setValidation('error');
    }
  };

  const blockDomainFirewall = async () => {
    const domain = domainInput.trim();
    if (!domain) return;
    try {
      const res = await window.electronAPI.firewallBlockDomain(domain);
      if (!res.success) alert(res.message || 'Failed to add firewall rule');
      await listFirewallRules();
    } catch (e) {
      alert('Firewall block failed: ' + e.message);
    }
  };

  const unblockDomainFirewall = async (domain) => {
    try {
      const res = await window.electronAPI.firewallUnblockDomain(domain);
      if (!res.success) alert(res.message || 'Failed to remove firewall rule');
      await listFirewallRules();
    } catch (e) {
      alert('Firewall unblock failed: ' + e.message);
    }
  };

  const listFirewallRules = async () => {
    try {
      const res = await window.electronAPI.firewallListRules();
      if (res.success) setRules(res.rules || []);
    } catch (e) {
      console.error('Failed to list firewall rules', e);
    }
  };

  const isLicenseActive = licenseStatus && licenseStatus.status === 'active';

  const updateSecurityStats = () => {
    setSecurityStats(prev => ({
      ...prev,
      threatsBlocked: prev.threatsBlocked + Math.floor(Math.random() * 3),
      safeConnections: prev.safeConnections + Math.floor(Math.random() * 10) + 1,
      lastScan: new Date().toLocaleString()
    }));
  };

  const loadLicenseStatus = async () => {
    try {
      const status = await window.electronAPI.getLicenseStatus();
      setLicenseStatus(status);
    } catch (e) {
      setLicenseStatus(null);
    }
  };

  const runSecurityScan = async () => {
    setScanning(true);
    
    try {
      // Simulate security scan
      await new Promise(resolve => setTimeout(resolve, 3000));
      
      const result = await window.electronAPI.writeLog('Security scan completed - System clean');
      
      setSecurityStats(prev => ({
        ...prev,
        lastScan: new Date().toLocaleString()
      }));
      
      // Show success notification
      alert('Security scan completed successfully. No threats detected.');
    } catch (error) {
      console.error('Security scan failed:', error);
      alert('Security scan failed. Please try again.');
    } finally {
      setScanning(false);
    }
  };

  const toggleSecuritySetting = async (setting) => {
    const newSettings = {
      ...securitySettings,
      [setting]: !securitySettings[setting]
    };
    
    setSecuritySettings(newSettings);
    
    try {
      await window.electronAPI.writeLog(`Security setting changed: ${setting} = ${newSettings[setting]}`);
    } catch (error) {
      console.error('Failed to log security setting change:', error);
    }
  };

  const getSeverityColor = (severity) => {
    switch (severity) {
      case 'critical': return 'text-red-600 bg-red-100';
      case 'high': return 'text-orange-600 bg-orange-100';
      case 'medium': return 'text-yellow-600 bg-yellow-100';
      default: return 'text-gray-600 bg-gray-100';
    }
  };

  const getTypeIcon = (type) => {
    switch (type) {
      case 'malware': return <ExclamationTriangleIcon className="h-4 w-4" />;
      case 'phishing': return <EyeSlashIcon className="h-4 w-4" />;
      case 'suspicious': return <GlobeAltIcon className="h-4 w-4" />;
      default: return <ShieldCheckIcon className="h-4 w-4" />;
    }
  };

  return (
    <div className="max-w-7xl mx-auto p-6">
      {/* Header */}
      <div className="mb-8">
        <div className="flex items-center justify-between">
          <div className="flex items-center">
            <div className="bg-gradient-to-r from-blue-500 to-purple-600 rounded-lg p-3 mr-4">
              <ShieldCheckIcon className="h-8 w-8 text-white" />
            </div>
            <div>
              <h1 className="text-3xl font-bold text-gray-900">Security Center</h1>
              <p className="text-gray-600">Advanced protection for your system and data</p>
            </div>
          </div>
          
          <div className="flex items-center space-x-4">
            <div className="text-center">
              <div className="flex items-center justify-center w-12 h-12 bg-green-100 rounded-full mb-2">
                <CheckCircleIcon className="h-6 w-6 text-green-600" />
              </div>
              <p className="text-xs text-gray-600">Protected</p>
            </div>
            
            <button
              onClick={runSecurityScan}
              disabled={scanning}
              className="flex items-center px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors disabled:opacity-50"
            >
              {scanning ? (
                <>
                  <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                  Scanning...
                </>
              ) : (
                <>
                  <BoltIcon className="h-4 w-4 mr-2" />
                  Quick Scan
                </>
              )}
            </button>
          </div>
        </div>
      </div>

      {/* Security Stats */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-6 mb-8">
        <div
          className="bg-gradient-to-r from-green-400 to-green-600 rounded-lg p-6 text-white cursor-pointer"
          role="button"
          tabIndex={0}
          onClick={() => recentBlocksRef.current?.scrollIntoView({ behavior: 'smooth' })}
          onKeyDown={(e) => { if (e.key === 'Enter' || e.key === ' ') recentBlocksRef.current?.scrollIntoView({ behavior: 'smooth' }); }}
        >
          <div className="flex items-center justify-between">
            <div>
              <p className="text-green-100 text-sm">Threats Blocked</p>
              <p className="text-3xl font-bold">{securityStats.threatsBlocked}</p>
            </div>
            <ShieldCheckIcon className="h-8 w-8 text-green-200" />
          </div>
        </div>

        <div
          className="bg-gradient-to-r from-blue-400 to-blue-600 rounded-lg p-6 text-white cursor-pointer"
          role="button"
          tabIndex={0}
          onClick={() => safeSectionRef.current?.scrollIntoView({ behavior: 'smooth' })}
          onKeyDown={(e) => { if (e.key === 'Enter' || e.key === ' ') safeSectionRef.current?.scrollIntoView({ behavior: 'smooth' }); }}
        >
          <div className="flex items-center justify-between">
            <div>
              <p className="text-blue-100 text-sm">Safe Connections</p>
              <p className="text-3xl font-bold">{securityStats.safeConnections}</p>
            </div>
            <GlobeAltIcon className="h-8 w-8 text-blue-200" />
          </div>
        </div>

        <div
          className="bg-gradient-to-r from-orange-400 to-red-500 rounded-lg p-6 text-white cursor-pointer"
          role="button"
          tabIndex={0}
          onClick={() => safeSectionRef.current?.scrollIntoView({ behavior: 'smooth' })}
          onKeyDown={(e) => { if (e.key === 'Enter' || e.key === ' ') safeSectionRef.current?.scrollIntoView({ behavior: 'smooth' }); }}
        >
          <div className="flex items-center justify-between">
            <div>
              <p className="text-orange-100 text-sm">Suspicious Sites</p>
              <p className="text-3xl font-bold">{securityStats.suspiciousWebsites}</p>
            </div>
            <ExclamationTriangleIcon className="h-8 w-8 text-orange-200" />
          </div>
        </div>

        <div
          className="bg-gradient-to-r from-purple-400 to-purple-600 rounded-lg p-6 text-white cursor-pointer"
          role="button"
          tabIndex={0}
          onClick={runSecurityScan}
          onKeyDown={(e) => { if (e.key === 'Enter' || e.key === ' ') runSecurityScan(); }}
        >
          <div className="flex items-center justify-between">
            <div>
              <p className="text-purple-100 text-sm">Last Scan</p>
              <p className="text-sm font-semibold">{securityStats.lastScan || 'Never'}</p>
            </div>
            <ClockIcon className="h-8 w-8 text-purple-200" />
          </div>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
        
        {/* Security Settings */}
        <div className="lg:col-span-2">
          <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6" ref={recentBlocksRef}>
            <div className="flex items-center mb-6">
              <Cog6ToothIcon className="h-6 w-6 text-blue-600 mr-2" />
              <h2 className="text-xl font-semibold text-gray-900">Protection Settings</h2>
            </div>
            
            <div className="space-y-4">
              {Object.entries(securitySettings).map(([key, value]) => (
                <div key={key} className="flex items-center justify-between p-4 bg-gray-50 rounded-lg">
                  <div>
                    <h3 className="font-medium text-gray-900 capitalize">
                      {key.replace(/([A-Z])/g, ' $1').replace(/^./, str => str.toUpperCase())}
                    </h3>
                    <p className="text-sm text-gray-600">
                      {key === 'realTimeProtection' && 'Monitor system activity in real-time'}
                      {key === 'websiteBlocking' && 'Block access to malicious websites'}
                      {key === 'malwareProtection' && 'Detect and prevent malware infections'}
                      {key === 'phishingProtection' && 'Protect against phishing attempts'}
                      {key === 'adultContentFilter' && 'Filter inappropriate content'}
                      {key === 'downloadScanning' && 'Scan downloaded files for threats'}
                    </p>
                  </div>
                  <label className="relative inline-flex items-center cursor-pointer">
                    <input
                      type="checkbox"
                      checked={value}
                      onChange={() => toggleSecuritySetting(key)}
                      className="sr-only peer"
                    />
                    <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                  </label>
                </div>
              ))}
            </div>
          </div>
        </div>

        {/* Recent Security Events */}
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
          <div className="flex items-center mb-6">
            <ChartBarIcon className="h-6 w-6 text-red-600 mr-2" />
            <h2 className="text-xl font-semibold text-gray-900">Recent Blocks</h2>
          </div>
          
          <div className="space-y-4">
            {recentBlocks.map((block) => (
              <div key={block.id} className="border border-gray-200 rounded-lg p-4">
                <div className="flex items-start justify-between">
                  <div className="flex items-start">
                    <div className={`p-2 rounded-full mr-3 ${getSeverityColor(block.severity)}`}>
                      {getTypeIcon(block.type)}
                    </div>
                    <div>
                      <h3 className="font-medium text-gray-900 capitalize">{block.type} Blocked</h3>
                      <p className="text-sm text-gray-600 break-all">{block.url}</p>
                      <p className="text-xs text-gray-500 mt-1">{block.timestamp}</p>
                    </div>
                  </div>
                  <span className={`px-2 py-1 text-xs font-medium rounded-full ${getSeverityColor(block.severity)}`}>
                    {block.severity}
                  </span>
                </div>
              </div>
            ))}
            
            {recentBlocks.length === 0 && (
              <div className="text-center py-8">
                <CheckCircleIcon className="h-12 w-12 text-green-500 mx-auto mb-4" />
                <p className="text-gray-600">No recent security events</p>
                <p className="text-sm text-gray-500">Your system is secure</p>
              </div>
            )}
          </div>
        </div>
      </div>

      {/* Emergency Contact */}
      <div className="mt-8 bg-gradient-to-r from-red-50 to-orange-50 border border-red-200 rounded-lg p-6">
        <div className="flex items-start">
          <ExclamationTriangleIcon className="h-6 w-6 text-red-600 mr-3 mt-1" />
          <div className="flex-1">
            <h3 className="text-lg font-semibold text-red-900 mb-2">Security Emergency Support</h3>
            <p className="text-red-800 mb-4">
              If you suspect your system has been compromised or need immediate security assistance, 
              contact our emergency support team immediately.
            </p>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="flex items-center">
                <PhoneIcon className="h-5 w-5 text-red-600 mr-2" />
                <div>
                  <p className="font-semibold text-red-900">Emergency Hotline</p>
                  <a 
                    href="tel:+1-888-401-3698" 
                    className="text-red-600 hover:text-red-700 font-semibold"
                  >
                    +1 (888) 401-3698
                  </a>
                </div>
              </div>
              
              <div className="flex items-center">
                <EnvelopeIcon className="h-5 w-5 text-red-600 mr-2" />
                <div>
                  <p className="font-semibold text-red-900">Security Team</p>
                  <a href="mailto:security@greeksquadusa.com" className="text-red-700 hover:text-red-800">
                    security@greeksquadusa.com
                  </a>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Safe Search and Firewall Controls */}
      <div className="mt-8 grid grid-cols-1 lg:grid-cols-2 gap-8">
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6" ref={safeSectionRef}>
          <div className="flex items-center mb-4">
            <GlobeAltIcon className="h-6 w-6 text-blue-600 mr-2" />
            <h2 className="text-lg font-semibold text-gray-900">Safe Search</h2>
          </div>
          <div className="flex items-center justify-between p-4 bg-gray-50 rounded-lg mb-4">
            <div>
              <p className="font-medium text-gray-900">Enable Safe Search</p>
              <p className="text-sm text-gray-600">Block known malicious domains and enforce safe browsing</p>
            </div>
            <label className="relative inline-flex items-center cursor-pointer">
              <input type="checkbox" className="sr-only peer" checked={safeSearchEnabled} onChange={toggleSafeSearch} />
              <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
            </label>
          </div>
          <div className="space-y-3">
            <div className="flex gap-2">
              <input value={domainInput} onChange={(e) => { setDomainInput(e.target.value); setValidation(null); }} placeholder="example.com" className="flex-1 px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500" />
              <button onClick={validateDomain} className="px-3 py-2 bg-gray-100 border border-gray-300 rounded-md text-sm">Validate</button>
              <button onClick={addToBlacklist} className="px-3 py-2 bg-red-600 text-white rounded-md text-sm">Add to Blacklist</button>
              <button onClick={addToWhitelist} className="px-3 py-2 bg-green-600 text-white rounded-md text-sm">Add to Whitelist</button>
            </div>
            {validation && (
              <div className={`text-sm ${validation === 'safe' ? 'text-green-600' : validation === 'blocked' ? 'text-red-600' : 'text-yellow-600'}`}>
                {validation === 'safe' ? 'Domain appears safe.' : validation === 'blocked' ? 'Domain is blocked by policy.' : 'Validation error.'}
              </div>
            )}
            <div className="grid grid-cols-2 gap-4">
              <div>
                <p className="text-sm font-medium text-gray-700 mb-1">Blacklist</p>
                <div className="border border-gray-200 rounded-md p-2 max-h-40 overflow-auto text-sm">
                  {blacklist.length === 0 && <p className="text-gray-500">Empty</p>}
                  {blacklist.map((d, i) => (<div key={i} className="text-gray-800">{d}</div>))}
                </div>
              </div>
              <div>
                <p className="text-sm font-medium text-gray-700 mb-1">Whitelist</p>
                <div className="border border-gray-200 rounded-md p-2 max-h-40 overflow-auto text-sm">
                  {whitelist.length === 0 && <p className="text-gray-500">Empty</p>}
                  {whitelist.map((d, i) => (<div key={i} className="text-gray-800">{d}</div>))}
                </div>
              </div>
            </div>
          </div>
        </div>
        <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6" ref={firewallSectionRef}>
          <div className="flex items-center mb-4">
            <ShieldCheckIcon className="h-6 w-6 text-purple-600 mr-2" />
            <h2 className="text-lg font-semibold text-gray-900">Firewall Controls</h2>
          </div>
          {!isLicenseActive && (
            <div className="mb-3 p-3 rounded border border-yellow-300 bg-yellow-50 text-yellow-800 text-sm">
              Firewall controls require an active license. Please activate your license in the License tab.
            </div>
          )}
          <div className="space-y-3">
            <div className="flex gap-2">
              <input value={domainInput} onChange={(e) => setDomainInput(e.target.value)} placeholder="example.com" className="flex-1 px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-purple-500" />
              <button onClick={blockDomainFirewall} disabled={!isLicenseActive} className="px-3 py-2 bg-purple-600 text-white rounded-md text-sm disabled:opacity-50">Block</button>
              <button onClick={listFirewallRules} disabled={!isLicenseActive} className="px-3 py-2 bg-gray-100 border border-gray-300 rounded-md text-sm disabled:opacity-50">Refresh Rules</button>
            </div>
            <div className="border border-gray-200 rounded-md p-2 max-h-48 overflow-auto text-sm">
              {rules.length === 0 && <p className="text-gray-500">No GDU rules</p>}
              {rules.map((r, i) => (
                <div key={i} className="flex items-center justify-between py-1">
                  <span className="text-gray-800">{r.DisplayName || r.displayName}</span>
                  <button onClick={() => unblockDomainFirewall(String(r.DisplayName || r.displayName).replace('GDU Block ', ''))} disabled={!isLicenseActive} className="px-2 py-1 text-xs bg-red-50 text-red-700 border border-red-200 rounded disabled:opacity-50">Unblock</button>
                </div>
              ))}
            </div>
            <p className="text-xs text-gray-500">Requires administrator privileges. Rules use Windows Defender Firewall outbound blocking.</p>
          </div>
        </div>
      </div>
    </div>
  );
};

export default SecurityCenter;
