import React, { useState, useEffect } from 'react';
import { 
  ShieldCheckIcon, 
  ExclamationTriangleIcon, 
  ArrowPathIcon,
  BellIcon,
  ComputerDesktopIcon,
  ClockIcon
} from '@heroicons/react/24/outline';

const RealTimeMonitor = () => {
  const [isMonitoring, setIsMonitoring] = useState(false);
  const [systemStatus, setSystemStatus] = useState(null);
  const [notifications, setNotifications] = useState([]);
  const [lastScan, setLastScan] = useState(null);
  const [scanProgress, setScanProgress] = useState(0);

  useEffect(() => {
    // Set up real-time event listeners
    window.electronAPI.onScanCompleted((event, scanResult) => {
      setSystemStatus(prev => ({ ...prev, ...scanResult }));
      setLastScan(new Date());
      addNotification('System scan completed', 'info');
    });

    window.electronAPI.onUpdatesAvailable((event, updates) => {
      addNotification(`${updates.available} updates available`, 'warning');
    });

    window.electronAPI.onSecurityAlert((event, alert) => {
      addNotification(`Security alert: ${alert.threats} threats detected`, 'critical');
    });

    window.electronAPI.onInitialScanComplete((event, scanResult) => {
      setSystemStatus(scanResult);
      setLastScan(new Date());
    });

    // Load initial system status
    loadSystemStatus();

    return () => {
      // Cleanup listeners
      window.electronAPI.removeAllListeners('scan-completed');
      window.electronAPI.removeAllListeners('updates-available');
      window.electronAPI.removeAllListeners('security-alert');
      window.electronAPI.removeAllListeners('initial-scan-complete');
    };
  }, []);

  const loadSystemStatus = async () => {
    try {
      const result = await window.electronAPI.getSystemStatus();
      if (result.success) {
        setSystemStatus(result.data);
        setIsMonitoring(result.data.isMonitoring);
        if (result.data.lastScan) {
          setLastScan(new Date(result.data.lastScan));
        }
      }
    } catch (error) {
      console.error('Failed to load system status:', error);
    }
  };

  const toggleMonitoring = async () => {
    try {
      if (isMonitoring) {
        await window.electronAPI.stopMonitoring();
        setIsMonitoring(false);
        addNotification('Real-time monitoring stopped', 'info');
      } else {
        await window.electronAPI.startMonitoring();
        setIsMonitoring(true);
        addNotification('Real-time monitoring started', 'success');
      }
    } catch (error) {
      addNotification('Failed to toggle monitoring', 'error');
    }
  };

  const runManualScan = async () => {
    setScanProgress(0);
    addNotification('Starting system scan...', 'info');
    
    // Simulate scan progress
    const progressInterval = setInterval(() => {
      setScanProgress(prev => {
        if (prev >= 90) {
          clearInterval(progressInterval);
          return 90;
        }
        return prev + 10;
      });
    }, 500);

    try {
      const result = await window.electronAPI.scanSystem();
      clearInterval(progressInterval);
      setScanProgress(100);
      
      if (result.success) {
        setSystemStatus(result.data);
        setLastScan(new Date());
        addNotification('System scan completed successfully', 'success');
      } else {
        addNotification('System scan failed', 'error');
      }
    } catch (error) {
      clearInterval(progressInterval);
      setScanProgress(0);
      addNotification('System scan error', 'error');
    }

    setTimeout(() => setScanProgress(0), 2000);
  };

  const addNotification = (message, type) => {
    const notification = {
      id: Date.now(),
      message,
      type,
      timestamp: new Date()
    };
    setNotifications(prev => [notification, ...prev.slice(0, 4)]);
  };

  const getStatusColor = (health) => {
    switch (health) {
      case 'good': return 'text-green-500';
      case 'warning': return 'text-yellow-500';
      case 'poor': return 'text-red-500';
      default: return 'text-gray-500';
    }
  };

  const getNotificationIcon = (type) => {
    switch (type) {
      case 'success': return '✅';
      case 'warning': return '⚠️';
      case 'critical': return '🚨';
      case 'error': return '❌';
      default: return 'ℹ️';
    }
  };

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="bg-gradient-to-r from-blue-600 to-purple-600 rounded-xl p-6 text-white">
        <div className="flex items-center justify-between">
          <div>
            <h2 className="text-2xl font-bold mb-2">Real-Time System Monitor</h2>
            <p className="text-blue-100">
              Continuous monitoring and instant notifications for your system
            </p>
          </div>
          <div className="text-right">
            <div className={`inline-flex items-center px-3 py-1 rounded-full text-sm font-medium ${
              isMonitoring ? 'bg-green-500 text-white' : 'bg-gray-500 text-white'
            }`}>
              <div className={`w-2 h-2 rounded-full mr-2 ${
                isMonitoring ? 'bg-white animate-pulse' : 'bg-gray-300'
              }`}></div>
              {isMonitoring ? 'MONITORING ACTIVE' : 'MONITORING INACTIVE'}
            </div>
          </div>
        </div>
      </div>

      {/* Control Panel */}
      <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
        <div className="bg-white rounded-xl shadow-lg p-6 border border-gray-200">
          <h3 className="text-lg font-semibold mb-4 flex items-center">
            <ArrowPathIcon className="w-5 h-5 mr-2 text-blue-600" />
            Monitoring Controls
          </h3>
          
          <div className="space-y-4">
            <button
              onClick={toggleMonitoring}
              className={`w-full py-3 px-4 rounded-lg font-medium transition-all duration-200 ${
                isMonitoring
                  ? 'bg-red-500 hover:bg-red-600 text-white'
                  : 'bg-green-500 hover:bg-green-600 text-white'
              }`}
            >
              {isMonitoring ? 'Stop Monitoring' : 'Start Monitoring'}
            </button>

            <button
              onClick={runManualScan}
              disabled={scanProgress > 0}
              className="w-full py-3 px-4 bg-blue-500 hover:bg-blue-600 disabled:bg-gray-400 text-white rounded-lg font-medium transition-all duration-200"
            >
              {scanProgress > 0 ? `Scanning... ${scanProgress}%` : 'Run Manual Scan'}
            </button>

            {scanProgress > 0 && (
              <div className="w-full bg-gray-200 rounded-full h-2">
                <div 
                  className="bg-blue-500 h-2 rounded-full transition-all duration-300"
                  style={{ width: `${scanProgress}%` }}
                ></div>
              </div>
            )}
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-lg p-6 border border-gray-200">
          <h3 className="text-lg font-semibold mb-4 flex items-center">
            <ClockIcon className="w-5 h-5 mr-2 text-purple-600" />
            Last Scan Info
          </h3>
          
          <div className="space-y-3">
            <div className="flex justify-between">
              <span className="text-gray-600">Last Scan:</span>
              <span className="font-medium">
                {lastScan ? lastScan.toLocaleString() : 'Never'}
              </span>
            </div>
            <div className="flex justify-between">
              <span className="text-gray-600">System Health:</span>
              <span className={`font-medium ${getStatusColor(systemStatus?.systemHealth)}`}>
                {systemStatus?.systemHealth?.toUpperCase() || 'UNKNOWN'}
              </span>
            </div>
            <div className="flex justify-between">
              <span className="text-gray-600">Monitoring:</span>
              <span className={`font-medium ${isMonitoring ? 'text-green-500' : 'text-gray-500'}`}>
                {isMonitoring ? 'ACTIVE' : 'INACTIVE'}
              </span>
            </div>
          </div>
        </div>
      </div>

      {/* System Status Overview */}
      {systemStatus && (
        <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
          <div className="bg-white rounded-xl shadow-lg p-4 border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Drivers Found</p>
                <p className="text-2xl font-bold text-blue-600">
                  {systemStatus.drivers?.length || 0}
                </p>
              </div>
              <ComputerDesktopIcon className="w-8 h-8 text-blue-500" />
            </div>
          </div>

          <div className="bg-white rounded-xl shadow-lg p-4 border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Windows Updates</p>
                <p className="text-2xl font-bold text-orange-600">
                  {systemStatus.windowsUpdates?.available || 0}
                </p>
              </div>
              <ArrowPathIcon className="w-8 h-8 text-orange-500" />
            </div>
          </div>

          <div className="bg-white rounded-xl shadow-lg p-4 border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">App Updates</p>
                <p className="text-2xl font-bold text-purple-600">
                  {systemStatus.appUpdates?.available || 0}
                </p>
              </div>
              <ArrowPathIcon className="w-8 h-8 text-purple-500" />
            </div>
          </div>

          <div className="bg-white rounded-xl shadow-lg p-4 border border-gray-200">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-sm text-gray-600">Security Threats</p>
                <p className="text-2xl font-bold text-red-600">
                  {systemStatus.security?.threats || 0}
                </p>
              </div>
              {systemStatus.security?.threats > 0 ? (
                <ExclamationTriangleIcon className="w-8 h-8 text-red-500" />
              ) : (
                <ShieldCheckIcon className="w-8 h-8 text-green-500" />
              )}
            </div>
          </div>
        </div>
      )}

      {/* Real-time Notifications */}
      <div className="bg-white rounded-xl shadow-lg p-6 border border-gray-200">
        <h3 className="text-lg font-semibold mb-4 flex items-center">
          <BellIcon className="w-5 h-5 mr-2 text-yellow-600" />
          Real-time Notifications
        </h3>
        
        <div className="space-y-3 max-h-64 overflow-y-auto">
          {notifications.length === 0 ? (
            <p className="text-gray-500 text-center py-4">No notifications yet</p>
          ) : (
            notifications.map((notification) => (
              <div
                key={notification.id}
                className="flex items-start space-x-3 p-3 bg-gray-50 rounded-lg"
              >
                <span className="text-lg">
                  {getNotificationIcon(notification.type)}
                </span>
                <div className="flex-1">
                  <p className="text-sm font-medium text-gray-900">
                    {notification.message}
                  </p>
                  <p className="text-xs text-gray-500">
                    {notification.timestamp.toLocaleTimeString()}
                  </p>
                </div>
              </div>
            ))
          )}
        </div>
      </div>

      {/* Performance Metrics */}
      {systemStatus?.performance && (
        <div className="bg-white rounded-xl shadow-lg p-6 border border-gray-200">
          <h3 className="text-lg font-semibold mb-4">System Performance</h3>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="text-center">
              <div className="text-3xl font-bold text-blue-600 mb-2">
                {systemStatus.performance.score}%
              </div>
              <div className="text-sm text-gray-600">Overall Score</div>
              <div className="w-full bg-gray-200 rounded-full h-2 mt-2">
                <div 
                  className="bg-blue-500 h-2 rounded-full"
                  style={{ width: `${systemStatus.performance.score}%` }}
                ></div>
              </div>
            </div>

            <div className="text-center">
              <div className="text-3xl font-bold text-green-600 mb-2">
                {systemStatus.performance.cpu}%
              </div>
              <div className="text-sm text-gray-600">CPU Performance</div>
              <div className="w-full bg-gray-200 rounded-full h-2 mt-2">
                <div 
                  className="bg-green-500 h-2 rounded-full"
                  style={{ width: `${systemStatus.performance.cpu}%` }}
                ></div>
              </div>
            </div>

            <div className="text-center">
              <div className="text-3xl font-bold text-purple-600 mb-2">
                {systemStatus.performance.memory}%
              </div>
              <div className="text-sm text-gray-600">Memory Available</div>
              <div className="w-full bg-gray-200 rounded-full h-2 mt-2">
                <div 
                  className="bg-purple-500 h-2 rounded-full"
                  style={{ width: `${systemStatus.performance.memory}%` }}
                ></div>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default RealTimeMonitor;
