import React, { useState, useEffect } from 'react';
import { 
  PrinterIcon, 
  CheckCircleIcon, 
  ExclamationTriangleIcon,
  XCircleIcon,
  ArrowPathIcon,
  DocumentTextIcon,
  WrenchScrewdriverIcon,
  InformationCircleIcon
} from '@heroicons/react/24/outline';

const PrinterManager = () => {
  const [printers, setPrinters] = useState([]);
  const [drivers, setDrivers] = useState([]);
  const [loading, setLoading] = useState(false);
  const [activeTab, setActiveTab] = useState('printers');
  const [troubleshootingPrinter, setTroubleshootingPrinter] = useState(null);

  useEffect(() => {
    loadPrinters();
    loadDrivers();
  }, []);

  const loadPrinters = async () => {
    setLoading(true);
    try {
      const result = await window.electronAPI.getPrinters();
      setPrinters(result);
      await window.electronAPI.writeLog(`Loaded ${result.length} printers`);
    } catch (error) {
      console.error('Failed to load printers:', error);
      await window.electronAPI.writeLog(`Failed to load printers: ${error.message}`);
    } finally {
      setLoading(false);
    }
  };

  const loadDrivers = async () => {
    try {
      const result = await window.electronAPI.getPrinterDrivers();
      setDrivers(result);
    } catch (error) {
      console.error('Failed to load drivers:', error);
    }
  };

  const restartSpooler = async () => {
    try {
      await window.electronAPI.restartPrintSpooler();
      await window.electronAPI.writeLog('Print spooler restarted');
      await loadPrinters();
    } catch (error) {
      console.error('Failed to restart spooler:', error);
      await window.electronAPI.writeLog(`Failed to restart spooler: ${error.message}`);
    }
  };

  const printTestPage = async (printerName) => {
    try {
      await window.electronAPI.printTestPage(printerName);
      await window.electronAPI.writeLog(`Test page sent to ${printerName}`);
    } catch (error) {
      console.error('Failed to print test page:', error);
      await window.electronAPI.writeLog(`Failed to print test page: ${error.message}`);
    }
  };

  const getStatusIcon = (status) => {
    switch (status?.toLowerCase()) {
      case 'idle':
      case 'ready':
        return <CheckCircleIcon className="h-5 w-5 text-success-500" />;
      case 'printing':
        return <ArrowPathIcon className="h-5 w-5 text-primary-500 animate-spin" />;
      case 'offline':
      case 'error':
        return <XCircleIcon className="h-5 w-5 text-danger-500" />;
      default:
        return <ExclamationTriangleIcon className="h-5 w-5 text-warning-500" />;
    }
  };

  const getStatusColor = (status) => {
    switch (status?.toLowerCase()) {
      case 'idle':
      case 'ready':
        return 'text-success-600 bg-success-100';
      case 'printing':
        return 'text-primary-600 bg-primary-100';
      case 'offline':
      case 'error':
        return 'text-danger-600 bg-danger-100';
      default:
        return 'text-warning-600 bg-warning-100';
    }
  };

  const troubleshootingSteps = {
    offline: [
      'Check if the printer is powered on',
      'Verify USB or network connection',
      'Restart the print spooler service',
      'Set printer as default if needed',
      'Update or reinstall printer drivers'
    ],
    driver: [
      'Download latest drivers from manufacturer website',
      'Uninstall current driver completely',
      'Restart computer',
      'Install new driver as administrator',
      'Test printer functionality'
    ],
    network: [
      'Check Wi-Fi connection on printer',
      'Verify printer and computer are on same network',
      'Restart router and printer',
      'Re-add printer using IP address',
      'Check firewall settings'
    ]
  };

  const TroubleshootingWizard = ({ printer, onClose }) => (
    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
      <div className="bg-white rounded-lg shadow-xl max-w-2xl w-full mx-4 max-h-96 overflow-y-auto">
        <div className="p-6 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">
            Troubleshooting: {printer.Name}
          </h3>
          <p className="text-gray-600 mt-1">Follow these steps to resolve common issues</p>
        </div>
        
        <div className="p-6 space-y-6">
          <div>
            <h4 className="font-medium text-gray-900 mb-3">Printer Offline Issues</h4>
            <ul className="space-y-2">
              {troubleshootingSteps.offline.map((step, index) => (
                <li key={index} className="flex items-start space-x-2">
                  <span className="flex-shrink-0 w-6 h-6 bg-primary-100 text-primary-600 rounded-full flex items-center justify-center text-sm font-medium">
                    {index + 1}
                  </span>
                  <span className="text-gray-700">{step}</span>
                </li>
              ))}
            </ul>
          </div>
          
          <div>
            <h4 className="font-medium text-gray-900 mb-3">Driver Issues</h4>
            <ul className="space-y-2">
              {troubleshootingSteps.driver.map((step, index) => (
                <li key={index} className="flex items-start space-x-2">
                  <span className="flex-shrink-0 w-6 h-6 bg-success-100 text-success-600 rounded-full flex items-center justify-center text-sm font-medium">
                    {index + 1}
                  </span>
                  <span className="text-gray-700">{step}</span>
                </li>
              ))}
            </ul>
          </div>
          
          <div>
            <h4 className="font-medium text-gray-900 mb-3">Network Printer Issues</h4>
            <ul className="space-y-2">
              {troubleshootingSteps.network.map((step, index) => (
                <li key={index} className="flex items-start space-x-2">
                  <span className="flex-shrink-0 w-6 h-6 bg-warning-100 text-warning-600 rounded-full flex items-center justify-center text-sm font-medium">
                    {index + 1}
                  </span>
                  <span className="text-gray-700">{step}</span>
                </li>
              ))}
            </ul>
          </div>
        </div>
        
        <div className="p-6 border-t border-gray-200 flex justify-end space-x-3">
          <button
            onClick={onClose}
            className="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50"
          >
            Close
          </button>
          <button
            onClick={() => printTestPage(printer.Name)}
            className="px-4 py-2 bg-primary-600 text-white rounded-md hover:bg-primary-700"
          >
            Print Test Page
          </button>
        </div>
      </div>
    </div>
  );

  return (
    <div className="p-6">
      <div className="mb-8">
        <h1 className="text-2xl font-bold text-gray-900">Printer Manager</h1>
        <p className="text-gray-600 mt-1">Monitor and manage your printers and drivers</p>
      </div>

      {/* Action Bar */}
      <div className="bg-white rounded-lg shadow mb-6">
        <div className="p-6 border-b border-gray-200">
          <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between space-y-4 sm:space-y-0">
            <div className="flex items-center space-x-4">
              <button
                onClick={loadPrinters}
                disabled={loading}
                className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-primary-600 hover:bg-primary-700 disabled:opacity-50"
              >
                <ArrowPathIcon className={`h-4 w-4 mr-2 ${loading ? 'animate-spin' : ''}`} />
                {loading ? 'Refreshing...' : 'Refresh Printers'}
              </button>
              
              <button
                onClick={restartSpooler}
                className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50"
              >
                <WrenchScrewdriverIcon className="h-4 w-4 mr-2" />
                Restart Spooler
              </button>
            </div>
          </div>
        </div>

        {/* Tabs */}
        <div className="border-b border-gray-200">
          <nav className="-mb-px flex space-x-8 px-6">
            <button
              onClick={() => setActiveTab('printers')}
              className={`py-2 px-1 border-b-2 font-medium text-sm ${
                activeTab === 'printers'
                  ? 'border-primary-500 text-primary-600'
                  : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
              }`}
            >
              Printers ({printers.length})
            </button>
            <button
              onClick={() => setActiveTab('drivers')}
              className={`py-2 px-1 border-b-2 font-medium text-sm ${
                activeTab === 'drivers'
                  ? 'border-primary-500 text-primary-600'
                  : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
              }`}
            >
              Drivers ({drivers.length})
            </button>
          </nav>
        </div>
      </div>

      {/* Content */}
      {activeTab === 'printers' && (
        <div className="space-y-4">
          {printers.length > 0 ? (
            printers.map((printer, index) => (
              <div key={index} className="bg-white rounded-lg shadow">
                <div className="p-6">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center space-x-4">
                      <PrinterIcon className="h-8 w-8 text-gray-400" />
                      <div>
                        <h3 className="text-lg font-medium text-gray-900 flex items-center">
                          {printer.Name}
                          {printer.Default && (
                            <span className="ml-2 px-2 py-1 text-xs font-medium bg-primary-100 text-primary-800 rounded-full">
                              Default
                            </span>
                          )}
                        </h3>
                        <p className="text-sm text-gray-500">Port: {printer.PortName}</p>
                        <p className="text-sm text-gray-500">Driver: {printer.DriverName}</p>
                      </div>
                    </div>
                    
                    <div className="flex items-center space-x-4">
                      <div className="flex items-center space-x-2">
                        {getStatusIcon(printer.Status)}
                        <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(printer.Status)}`}>
                          {printer.Status || 'Unknown'}
                        </span>
                      </div>
                      
                      <div className="flex space-x-2">
                        <button
                          onClick={() => printTestPage(printer.Name)}
                          className="inline-flex items-center px-3 py-1 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50"
                        >
                          <DocumentTextIcon className="h-4 w-4 mr-1" />
                          Test Page
                        </button>
                        <button
                          onClick={() => setTroubleshootingPrinter(printer)}
                          className="inline-flex items-center px-3 py-1 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50"
                        >
                          <InformationCircleIcon className="h-4 w-4 mr-1" />
                          Troubleshoot
                        </button>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            ))
          ) : (
            <div className="bg-white rounded-lg shadow p-12 text-center">
              <PrinterIcon className="h-12 w-12 text-gray-400 mx-auto mb-4" />
              <h3 className="text-lg font-medium text-gray-900 mb-2">No printers found</h3>
              <p className="text-gray-500">Connect a printer to get started.</p>
            </div>
          )}
        </div>
      )}

      {activeTab === 'drivers' && (
        <div className="bg-white rounded-lg shadow">
          <div className="divide-y divide-gray-200">
            {drivers.length > 0 ? (
              drivers.map((driver, index) => (
                <div key={index} className="p-6">
                  <div className="flex items-center justify-between">
                    <div>
                      <h3 className="text-lg font-medium text-gray-900">{driver.Name}</h3>
                      <p className="text-sm text-gray-500">Manufacturer: {driver.Manufacturer}</p>
                      <p className="text-sm text-gray-500">Version: {driver.Version}</p>
                      <p className="text-sm text-gray-500">Date: {driver.DriverDate}</p>
                    </div>
                    
                    <div className="flex space-x-2">
                      <span className="px-2 py-1 text-xs font-medium bg-success-100 text-success-800 rounded-full">
                        Installed
                      </span>
                    </div>
                  </div>
                </div>
              ))
            ) : (
              <div className="p-12 text-center">
                <h3 className="text-lg font-medium text-gray-900 mb-2">No drivers found</h3>
                <p className="text-gray-500">Install printer drivers to see them here.</p>
              </div>
            )}
          </div>
        </div>
      )}

      {/* Troubleshooting Modal */}
      {troubleshootingPrinter && (
        <TroubleshootingWizard
          printer={troubleshootingPrinter}
          onClose={() => setTroubleshootingPrinter(null)}
        />
      )}
    </div>
  );
};

export default PrinterManager;
