import React, { useState, useEffect } from 'react';
import { 
  DocumentTextIcon, 
  ArrowPathIcon, 
  TrashIcon,
  FunnelIcon,
  MagnifyingGlassIcon,
  ExclamationTriangleIcon,
  InformationCircleIcon,
  XCircleIcon
} from '@heroicons/react/24/outline';

const Logs = () => {
  const [logs, setLogs] = useState([]);
  const [filteredLogs, setFilteredLogs] = useState([]);
  const [loading, setLoading] = useState(false);
  const [filter, setFilter] = useState('all');
  const [searchTerm, setSearchTerm] = useState('');

  useEffect(() => {
    loadLogs();
  }, []);

  useEffect(() => {
    filterLogs();
  }, [logs, filter, searchTerm]);

  const loadLogs = async () => {
    setLoading(true);
    try {
      const result = await window.electronAPI.readLogs();
      const parsedLogs = result.logs.map(log => {
        const match = log.match(/^\[(.*?)\] (.*)$/);
        if (match) {
          const [, timestamp, message] = match;
          return {
            timestamp: new Date(timestamp),
            message: message.trim(),
            level: getLogLevel(message),
            id: Math.random().toString(36).substr(2, 9)
          };
        }
        return {
          timestamp: new Date(),
          message: log,
          level: 'info',
          id: Math.random().toString(36).substr(2, 9)
        };
      });
      setLogs(parsedLogs);
    } catch (error) {
      console.error('Failed to load logs:', error);
    } finally {
      setLoading(false);
    }
  };

  const getLogLevel = (message) => {
    const lowerMessage = message.toLowerCase();
    if (lowerMessage.includes('error') || lowerMessage.includes('failed')) {
      return 'error';
    } else if (lowerMessage.includes('warning') || lowerMessage.includes('warn')) {
      return 'warning';
    } else if (lowerMessage.includes('success') || lowerMessage.includes('completed')) {
      return 'success';
    }
    return 'info';
  };

  const filterLogs = () => {
    let filtered = logs;

    // Filter by level
    if (filter !== 'all') {
      filtered = filtered.filter(log => log.level === filter);
    }

    // Filter by search term
    if (searchTerm) {
      filtered = filtered.filter(log => 
        log.message.toLowerCase().includes(searchTerm.toLowerCase())
      );
    }

    setFilteredLogs(filtered);
  };

  const clearLogs = async () => {
    if (window.confirm('Are you sure you want to clear all logs? This action cannot be undone.')) {
      try {
        // Write a clear log entry
        await window.electronAPI.writeLog('Logs cleared by user');
        setLogs([]);
        setFilteredLogs([]);
      } catch (error) {
        console.error('Failed to clear logs:', error);
      }
    }
  };

  const exportLogs = () => {
    const logText = filteredLogs.map(log => 
      `[${log.timestamp.toISOString()}] ${log.message}`
    ).join('\n');
    
    const blob = new Blob([logText], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `pc-printer-assistant-logs-${new Date().toISOString().split('T')[0]}.txt`;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  };

  const getLogIcon = (level) => {
    switch (level) {
      case 'error':
        return <XCircleIcon className="h-5 w-5 text-danger-500" />;
      case 'warning':
        return <ExclamationTriangleIcon className="h-5 w-5 text-warning-500" />;
      case 'success':
        return <InformationCircleIcon className="h-5 w-5 text-success-500" />;
      default:
        return <InformationCircleIcon className="h-5 w-5 text-primary-500" />;
    }
  };

  const getLogColor = (level) => {
    switch (level) {
      case 'error':
        return 'border-l-danger-500 bg-danger-50';
      case 'warning':
        return 'border-l-warning-500 bg-warning-50';
      case 'success':
        return 'border-l-success-500 bg-success-50';
      default:
        return 'border-l-primary-500 bg-primary-50';
    }
  };

  const logCounts = {
    all: logs.length,
    error: logs.filter(log => log.level === 'error').length,
    warning: logs.filter(log => log.level === 'warning').length,
    success: logs.filter(log => log.level === 'success').length,
    info: logs.filter(log => log.level === 'info').length
  };

  return (
    <div className="p-6">
      <div className="mb-8">
        <h1 className="text-2xl font-bold text-gray-900">Activity Logs</h1>
        <p className="text-gray-600 mt-1">View system activity and troubleshoot issues</p>
      </div>

      {/* Controls */}
      <div className="bg-white rounded-lg shadow mb-6">
        <div className="p-6 border-b border-gray-200">
          <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between space-y-4 sm:space-y-0">
            <div className="flex items-center space-x-4">
              <button
                onClick={loadLogs}
                disabled={loading}
                className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-primary-600 hover:bg-primary-700 disabled:opacity-50"
              >
                <ArrowPathIcon className={`h-4 w-4 mr-2 ${loading ? 'animate-spin' : ''}`} />
                {loading ? 'Loading...' : 'Refresh'}
              </button>
              
              <button
                onClick={exportLogs}
                className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50"
              >
                <DocumentTextIcon className="h-4 w-4 mr-2" />
                Export
              </button>
              
              <button
                onClick={clearLogs}
                className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-danger-700 bg-white hover:bg-danger-50"
              >
                <TrashIcon className="h-4 w-4 mr-2" />
                Clear Logs
              </button>
            </div>
            
            <div className="text-sm text-gray-500">
              Showing {filteredLogs.length} of {logs.length} entries
            </div>
          </div>
        </div>

        {/* Filters */}
        <div className="p-6 bg-gray-50 border-b border-gray-200">
          <div className="flex flex-col sm:flex-row sm:items-center space-y-4 sm:space-y-0 sm:space-x-4">
            <div className="flex items-center space-x-2">
              <FunnelIcon className="h-5 w-5 text-gray-400" />
              <span className="text-sm font-medium text-gray-700">Filter:</span>
              <select
                value={filter}
                onChange={(e) => setFilter(e.target.value)}
                className="rounded-md border-gray-300 py-1 pl-3 pr-8 text-sm focus:border-primary-500 focus:outline-none focus:ring-primary-500"
              >
                <option value="all">All ({logCounts.all})</option>
                <option value="error">Errors ({logCounts.error})</option>
                <option value="warning">Warnings ({logCounts.warning})</option>
                <option value="success">Success ({logCounts.success})</option>
                <option value="info">Info ({logCounts.info})</option>
              </select>
            </div>
            
            <div className="flex-1 max-w-md">
              <div className="relative">
                <MagnifyingGlassIcon className="absolute left-3 top-1/2 transform -translate-y-1/2 h-4 w-4 text-gray-400" />
                <input
                  type="text"
                  placeholder="Search logs..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="block w-full pl-10 pr-3 py-2 border border-gray-300 rounded-md text-sm placeholder-gray-500 focus:outline-none focus:ring-1 focus:ring-primary-500 focus:border-primary-500"
                />
              </div>
            </div>
          </div>
        </div>
      </div>

      {/* Logs List */}
      <div className="bg-white rounded-lg shadow">
        {filteredLogs.length > 0 ? (
          <div className="divide-y divide-gray-200 max-h-96 overflow-y-auto">
            {filteredLogs.map((log) => (
              <div key={log.id} className={`p-4 border-l-4 ${getLogColor(log.level)}`}>
                <div className="flex items-start space-x-3">
                  {getLogIcon(log.level)}
                  <div className="flex-1 min-w-0">
                    <p className="text-sm text-gray-900 break-words">{log.message}</p>
                    <p className="text-xs text-gray-500 mt-1">
                      {log.timestamp.toLocaleString()}
                    </p>
                  </div>
                </div>
              </div>
            ))}
          </div>
        ) : (
          <div className="p-12 text-center">
            <DocumentTextIcon className="h-12 w-12 text-gray-400 mx-auto mb-4" />
            <h3 className="text-lg font-medium text-gray-900 mb-2">
              {logs.length === 0 ? 'No logs available' : 'No logs match your filters'}
            </h3>
            <p className="text-gray-500">
              {logs.length === 0 
                ? 'Activity logs will appear here as you use the application.'
                : 'Try adjusting your search terms or filters.'
              }
            </p>
          </div>
        )}
      </div>

      {/* Log Statistics */}
      {logs.length > 0 && (
        <div className="bg-white rounded-lg shadow mt-6">
          <div className="p-6 border-b border-gray-200">
            <h2 className="text-lg font-semibold text-gray-900">Log Statistics</h2>
          </div>
          <div className="p-6">
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              <div className="text-center">
                <div className="text-2xl font-bold text-danger-600">{logCounts.error}</div>
                <div className="text-sm text-gray-500">Errors</div>
              </div>
              <div className="text-center">
                <div className="text-2xl font-bold text-warning-600">{logCounts.warning}</div>
                <div className="text-sm text-gray-500">Warnings</div>
              </div>
              <div className="text-center">
                <div className="text-2xl font-bold text-success-600">{logCounts.success}</div>
                <div className="text-sm text-gray-500">Success</div>
              </div>
              <div className="text-center">
                <div className="text-2xl font-bold text-primary-600">{logCounts.info}</div>
                <div className="text-sm text-gray-500">Info</div>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default Logs;
