import React, { useState, useEffect } from 'react';
import { 
  KeyIcon, 
  CheckCircleIcon, 
  ExclamationTriangleIcon,
  XCircleIcon,
  ClockIcon,
  ShieldCheckIcon,
  GlobeAltIcon
} from '@heroicons/react/24/outline';

const LicenseActivation = ({ onLicenseActivated }) => {
  const [licenseStatus, setLicenseStatus] = useState(null);
  const [licenseKey, setLicenseKey] = useState('');
  const [email, setEmail] = useState('');
  const [loading, setLoading] = useState(false);
  const [message, setMessage] = useState('');
  const [showGenerator, setShowGenerator] = useState(false);
  const [activationAudit, setActivationAudit] = useState(null);

  useEffect(() => {
    checkLicenseStatus();
  }, []);

  const checkLicenseStatus = async () => {
    try {
      const status = await window.electronAPI.getLicenseStatus();
      setLicenseStatus(status);
    } catch (error) {
      console.error('Failed to check license status:', error);
    }
  };

  const activateLicense = async () => {
    if (!licenseKey.trim() || !email.trim()) {
      setMessage('Please enter both license key and email address');
      return;
    }

    setLoading(true);
    setMessage('');

    try {
      const result = await window.electronAPI.activateLicense(licenseKey.trim(), email.trim());
      
      if (result.success) {
        setMessage('License activated successfully!');
        setActivationAudit(result.audit || null);
        await checkLicenseStatus();
        if (onLicenseActivated) onLicenseActivated();
        await window.electronAPI.writeLog(`License activated for ${email}`);
      } else {
        setMessage(result.message);
      }
    } catch (error) {
      setMessage('Error activating license');
      console.error('License activation error:', error);
    } finally {
      setLoading(false);
    }
  };

  const deactivateLicense = async () => {
    if (!window.confirm('Are you sure you want to deactivate your license?')) return;

    setLoading(true);
    try {
      const result = await window.electronAPI.deactivateLicense();
      if (result.success) {
        setMessage('License deactivated');
        await checkLicenseStatus();
        await window.electronAPI.writeLog('License deactivated');
      } else {
        setMessage(result.message);
      }
    } catch (error) {
      setMessage('Error deactivating license');
    } finally {
      setLoading(false);
    }
  };

  const generateLicenseKey = async () => {
    if (!email.trim()) {
      setMessage('Please enter an email address');
      return;
    }

    setLoading(true);
    try {
      const result = await window.electronAPI.generateLicenseKey(email.trim(), 1);
      if (result.success) {
        setLicenseKey(result.licenseKey);
        setMessage(`License key generated! Expires: ${new Date(result.expiryDate).toLocaleDateString()}`);
      } else {
        setMessage(result.message);
      }
    } catch (error) {
      setMessage('Error generating license key');
    } finally {
      setLoading(false);
    }
  };

  const getStatusIcon = (status) => {
    switch (status) {
      case 'active':
        return <CheckCircleIcon className="h-8 w-8 text-green-500" />;
      case 'trial':
        return <ClockIcon className="h-8 w-8 text-blue-500" />;
      case 'expired':
        return <XCircleIcon className="h-8 w-8 text-red-500" />;
      case 'invalid':
        return <ExclamationTriangleIcon className="h-8 w-8 text-yellow-500" />;
      default:
        return <KeyIcon className="h-8 w-8 text-gray-500" />;
    }
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'active':
        return 'text-green-600 bg-green-100';
      case 'trial':
        return 'text-blue-600 bg-blue-100';
      case 'expired':
        return 'text-red-600 bg-red-100';
      case 'invalid':
        return 'text-yellow-600 bg-yellow-100';
      default:
        return 'text-gray-600 bg-gray-100';
    }
  };

  if (!licenseStatus) {
    return (
      <div className="flex items-center justify-center h-64">
        <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  return (
    <div className="max-w-2xl mx-auto">
      {/* License Status Card */}
      <div className="bg-white rounded-lg shadow mb-6">
        <div className="p-6 border-b border-gray-200">
          <div className="flex items-center justify-between">
            <h2 className="text-lg font-semibold text-gray-900 flex items-center">
              <ShieldCheckIcon className="h-6 w-6 mr-2" />
              License Status
            </h2>
            <div className="flex items-center space-x-2 text-sm text-blue-600">
              <GlobeAltIcon className="h-4 w-4" />
              <span>Greek Squad USA</span>
            </div>
          </div>
        </div>
        
        <div className="p-6">
          <div className="flex items-center space-x-4 mb-4">
            {getStatusIcon(licenseStatus.status)}
            <div>
              <span className={`inline-flex items-center px-3 py-1 rounded-full text-sm font-medium ${getStatusColor(licenseStatus.status)}`}>
                {licenseStatus.status.charAt(0).toUpperCase() + licenseStatus.status.slice(1)}
              </span>
              <p className="text-gray-600 mt-1">{licenseStatus.message}</p>
            </div>
          </div>

          {licenseStatus.status === 'active' && (
            <div className="bg-green-50 rounded-lg p-4 mb-4">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                <div>
                  <span className="font-medium text-gray-700">Licensed to:</span>
                  <span className="ml-2 text-gray-900">{licenseStatus.email}</span>
                </div>
                <div>
                  <span className="font-medium text-gray-700">Expires:</span>
                  <span className="ml-2 text-gray-900">
                    {new Date(licenseStatus.expiryDate).toLocaleDateString()}
                  </span>
                </div>
              </div>
            </div>
          )}

          {licenseStatus.status === 'trial' && (
            <div className="bg-blue-50 rounded-lg p-4 mb-4">
              <p className="text-blue-800 font-medium">
                Trial Period: {licenseStatus.daysRemaining} days remaining
              </p>
              <p className="text-blue-600 text-sm mt-1">
                Purchase a license to continue using Greek Driver Updater after the trial period.
              </p>
            </div>
          )}

          {(licenseStatus.status === 'expired' || licenseStatus.status === 'invalid') && (
            <div className="bg-red-50 rounded-lg p-4 mb-4">
              <p className="text-red-800 font-medium">License Required</p>
              <p className="text-red-600 text-sm mt-1">
                Please activate a valid license to continue using Greek Driver Updater.
              </p>
            </div>
          )}
        </div>
      </div>

      {/* License Activation Form */}
      {(licenseStatus.status !== 'active') && (
        <div className="bg-white rounded-lg shadow mb-6">
          <div className="p-6 border-b border-gray-200">
            <h3 className="text-lg font-semibold text-gray-900">Activate License</h3>
            <p className="text-gray-600 mt-1">Enter your license key to activate Greek Driver Updater</p>
          </div>
          
          <div className="p-6 space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                Email Address
              </label>
              <input
                type="email"
                value={email}
                onChange={(e) => setEmail(e.target.value)}
                placeholder="Enter your email address"
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
              />
            </div>
            
            <div>
              <label className="block text-sm font-medium text-gray-700 mb-2">
                License Key
              </label>
              <input
                type="text"
                value={licenseKey}
                onChange={(e) => setLicenseKey(e.target.value.toUpperCase())}
                placeholder="GDU-XXXXXXXX-XXXXXXXX-XXXXXXXX"
                className="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-blue-500 font-mono"
              />
            </div>
            
            {message && (
              <div className={`p-3 rounded-md ${
                message.includes('successfully') || message.includes('generated') 
                  ? 'bg-green-100 text-green-800' 
                  : 'bg-red-100 text-red-800'
              }`}>
                {message}
              </div>
            )}
            
            <div className="flex space-x-3">
              <button
                onClick={activateLicense}
                disabled={loading}
                className="flex-1 inline-flex items-center justify-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 disabled:opacity-50"
              >
                {loading ? (
                  <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                ) : (
                  <KeyIcon className="h-4 w-4 mr-2" />
                )}
                Activate License
              </button>
              
              <button
                onClick={() => setShowGenerator(!showGenerator)}
                className="px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50"
              >
                {showGenerator ? 'Hide' : 'Generate'} Key
              </button>
            </div>
          </div>
        </div>
      )}

      {/* License Key Generator (for demo purposes) */}
      {showGenerator && (
        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mb-6">
          <div className="flex items-center mb-3">
            <ExclamationTriangleIcon className="h-5 w-5 text-yellow-600 mr-2" />
            <h4 className="font-medium text-yellow-800">Demo License Generator</h4>
          </div>
          <p className="text-yellow-700 text-sm mb-3">
            This is for demonstration purposes only. In production, license keys would be generated by Greek Squad USA.
          </p>
          <button
            onClick={generateLicenseKey}
            disabled={loading || !email.trim()}
            className="inline-flex items-center px-3 py-2 border border-yellow-300 text-sm font-medium rounded-md text-yellow-800 bg-yellow-100 hover:bg-yellow-200 disabled:opacity-50"
          >
            Generate Demo Key
          </button>
        </div>
      )}

      {/* Active License Management */}
      {licenseStatus.status === 'active' && (
        <div className="bg-white rounded-lg shadow">
          <div className="p-6 border-b border-gray-200">
            <h3 className="text-lg font-semibold text-gray-900">License Management</h3>
          </div>
          
          <div className="p-6">
            <div className="flex items-center justify-between">
              <div>
                <p className="text-gray-900 font-medium">Your license is active and valid</p>
                <p className="text-gray-600 text-sm">
                  Contact Greek Squad USA for license transfers or renewals
                </p>
              </div>
              <button
                onClick={deactivateLicense}
                disabled={loading}
                className="inline-flex items-center px-4 py-2 border border-red-300 text-sm font-medium rounded-md text-red-700 bg-white hover:bg-red-50 disabled:opacity-50"
              >
                Deactivate License
              </button>
            </div>
            
            <div className="mt-4 pt-4 border-t border-gray-200">
              {activationAudit && (
                <div className="mb-4 text-sm">
                  <p className="font-medium text-gray-700 mb-1">Activation Security Audit</p>
                  <ul className="list-disc list-inside text-gray-600">
                    <li>Admin privileges: <span className={activationAudit.admin ? 'text-green-700' : 'text-yellow-700'}>{activationAudit.admin ? 'Yes' : 'No'}</span></li>
                    <li>Windows Defender issues: <span className={activationAudit.defenderIssues ? 'text-red-700' : 'text-green-700'}>{activationAudit.defenderIssues ? 'Detected' : 'None'}</span></li>
                    <li>Firewall enabled: <span className={activationAudit.firewallEnabled ? 'text-green-700' : 'text-yellow-700'}>{String(activationAudit.firewallEnabled)}</span></li>
                    <li>Audit time: <span className="text-gray-700">{new Date(activationAudit.timestamp).toLocaleString()}</span></li>
                  </ul>
                </div>
              )}
              <a 
                href="https://www.greeksquadusa.com" 
                target="_blank" 
                rel="noopener noreferrer"
                className="text-blue-600 hover:text-blue-700 text-sm"
              >
                Visit www.greeksquadusa.com for support →
              </a>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default LicenseActivation;
