import React, { useState, useEffect } from 'react';
import { 
  ArrowDownTrayIcon, 
  CheckCircleIcon, 
  ExclamationTriangleIcon,
  ArrowPathIcon,
  ClockIcon,
  GlobeAltIcon,
  ComputerDesktopIcon
} from '@heroicons/react/24/outline';

const DriverUpdates = ({ onUpdateComplete }) => {
  const [drivers, setDrivers] = useState([]);
  const [loading, setLoading] = useState(false);
  const [downloading, setDownloading] = useState(new Set());
  const [lastCheck, setLastCheck] = useState(null);
  const [downloadHistory, setDownloadHistory] = useState([]);

  useEffect(() => {
    checkForDriverUpdates();
  }, []);

  const checkForDriverUpdates = async () => {
    setLoading(true);
    try {
      const result = await window.electronAPI.checkDriverUpdates();
      setDrivers(result.drivers);
      setLastCheck(new Date().toLocaleString());
      await window.electronAPI.writeLog(`Checked for driver updates: ${result.drivers.filter(d => d.hasUpdate).length} available`);
    } catch (error) {
      console.error('Failed to check for driver updates:', error);
      await window.electronAPI.writeLog(`Failed to check for driver updates: ${error.message}`);
    } finally {
      setLoading(false);
    }
  };

  const downloadDriver = async (driver) => {
    setDownloading(prev => new Set([...prev, driver.name]));
    
    try {
      const result = await window.electronAPI.downloadDriver(driver);
      await window.electronAPI.writeLog(`Opened driver download for ${driver.name} (${driver.manufacturer})`);
      
      setDownloadHistory(prev => [...prev, {
        name: driver.name,
        manufacturer: driver.manufacturer,
        timestamp: new Date().toLocaleString(),
        url: result.url
      }]);
      
      if (onUpdateComplete) onUpdateComplete();
    } catch (error) {
      console.error('Failed to download driver:', error);
      await window.electronAPI.writeLog(`Failed to download driver for ${driver.name}: ${error.message}`);
    } finally {
      setDownloading(prev => {
        const newSet = new Set(prev);
        newSet.delete(driver.name);
        return newSet;
      });
    }
  };

  const getBrandIcon = (brand) => {
    const brandColors = {
      'hp': 'text-blue-600',
      'canon': 'text-red-600',
      'epson': 'text-blue-800',
      'brother': 'text-orange-600',
      'samsung': 'text-blue-700',
      'lexmark': 'text-red-700',
      'xerox': 'text-blue-600',
      'ricoh': 'text-green-600'
    };
    
    const key = (brand || '').toLowerCase();
    return (
      <div className={`w-12 h-12 rounded-lg flex items-center justify-center bg-gray-100 ${brandColors[key] || 'text-gray-600'}`}>
        <ComputerDesktopIcon className="h-6 w-6" />
      </div>
    );
  };

  const availableDrivers = drivers.filter(d => d.hasUpdate);
  const unavailableDrivers = drivers.filter(d => !d.hasUpdate);

  return (
    <div className="p-6">
      <div className="mb-8">
        <h1 className="text-2xl font-bold text-gray-900">Driver Updates</h1>
        <p className="text-gray-600 mt-1">Automatically detect and download the latest printer drivers</p>
      </div>

      {/* Action Bar */}
      <div className="bg-white rounded-lg shadow mb-6">
        <div className="p-6 border-b border-gray-200">
          <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between space-y-4 sm:space-y-0">
            <div className="flex items-center space-x-4">
              <button
                onClick={checkForDriverUpdates}
                disabled={loading}
                className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 disabled:opacity-50"
              >
                <ArrowPathIcon className={`h-4 w-4 mr-2 ${loading ? 'animate-spin' : ''}`} />
                {loading ? 'Scanning...' : 'Scan for Drivers'}
              </button>
              
              <div className="text-sm text-gray-600">
                <span className="font-medium">{availableDrivers.length}</span> driver{availableDrivers.length !== 1 ? 's' : ''} available
              </div>
            </div>
            
            {lastCheck && (
              <div className="flex items-center text-sm text-gray-500">
                <ClockIcon className="h-4 w-4 mr-1" />
                Last scan: {lastCheck}
              </div>
            )}
          </div>
        </div>

        {/* Greek Squad Branding */}
        <div className="px-6 py-3 bg-blue-50 border-b border-blue-200">
          <div className="flex items-center justify-between">
            <div className="flex items-center">
              <GlobeAltIcon className="h-5 w-5 text-blue-600 mr-2" />
              <span className="text-sm font-medium text-blue-900">Powered by Greek Squad USA</span>
            </div>
            <a 
              href="https://www.greeksquadusa.com" 
              target="_blank" 
              rel="noopener noreferrer"
              className="text-sm text-blue-600 hover:text-blue-700"
            >
              Visit www.greeksquadusa.com
            </a>
          </div>
        </div>
      </div>

      {/* Available Drivers */}
      {availableDrivers.length > 0 && (
        <div className="bg-white rounded-lg shadow mb-6">
          <div className="p-6 border-b border-gray-200">
            <h2 className="text-lg font-semibold text-gray-900 flex items-center">
              <ArrowDownTrayIcon className="h-5 w-5 mr-2 text-green-600" />
              Available Driver Updates ({availableDrivers.length})
            </h2>
            <p className="text-sm text-gray-600 mt-1">Click download to get the latest drivers for your printers</p>
          </div>
          <div className="divide-y divide-gray-200">
            {availableDrivers.map((driver, index) => (
              <div
                key={index}
                className="p-6 hover:bg-gray-50 cursor-pointer"
                role="button"
                tabIndex={0}
                onClick={() => !downloading.has(driver.name) && downloadDriver(driver)}
                onKeyDown={(e) => {
                  if ((e.key === 'Enter' || e.key === ' ') && !downloading.has(driver.name)) {
                    e.preventDefault();
                    downloadDriver(driver);
                  }
                }}
              >
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-4">
                    {getBrandIcon(driver.brand)}
                    <div>
                      <h3 className="text-lg font-medium text-gray-900">{driver.name}</h3>
                      <p className="text-sm text-gray-500">Manufacturer: {driver.manufacturer}</p>
                      <p className="text-sm text-gray-500">Current Driver: {driver.driver}</p>
                      <div className="flex items-center mt-1">
                        <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800">
                          <CheckCircleIcon className="h-3 w-3 mr-1" />
                          Driver Available
                        </span>
                      </div>
                    </div>
                  </div>
                  
                  <div className="flex items-center space-x-3">
                    <button
                      onClick={() => downloadDriver(driver)}
                      disabled={downloading.has(driver.name)}
                      className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-green-600 hover:bg-green-700 disabled:opacity-50"
                    >
                      {downloading.has(driver.name) ? (
                        <>
                          <ArrowPathIcon className="h-4 w-4 mr-2 animate-spin" />
                          Opening...
                        </>
                      ) : (
                        <>
                          <ArrowDownTrayIcon className="h-4 w-4 mr-2" />
                          Download Driver
                        </>
                      )}
                    </button>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* Unavailable Drivers */}
      {unavailableDrivers.length > 0 && (
        <div className="bg-white rounded-lg shadow mb-6">
          <div className="p-6 border-b border-gray-200">
            <h2 className="text-lg font-semibold text-gray-900 flex items-center">
              <ExclamationTriangleIcon className="h-5 w-5 mr-2 text-yellow-600" />
              Unsupported Printers ({unavailableDrivers.length})
            </h2>
            <p className="text-sm text-gray-600 mt-1">These printers are not supported for automatic driver updates</p>
          </div>
          <div className="divide-y divide-gray-200">
            {unavailableDrivers.map((driver, index) => (
              <div key={index} className="p-6">
                <div className="flex items-center space-x-4">
                  <div className="w-12 h-12 rounded-lg flex items-center justify-center bg-gray-100 text-gray-400">
                    <ComputerDesktopIcon className="h-6 w-6" />
                  </div>
                  <div>
                    <h3 className="text-lg font-medium text-gray-900">{driver.name}</h3>
                    <p className="text-sm text-gray-500">Manufacturer: {driver.manufacturer || 'Unknown'}</p>
                    <p className="text-sm text-gray-500">Current Driver: {driver.driver}</p>
                    <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-gray-100 text-gray-800 mt-1">
                      Manual Download Required
                    </span>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* No Printers Found */}
      {drivers.length === 0 && !loading && (
        <div className="bg-white rounded-lg shadow p-12 text-center">
          <ComputerDesktopIcon className="h-12 w-12 text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">No printers detected</h3>
          <p className="text-gray-500">Connect a printer to your computer to check for driver updates.</p>
        </div>
      )}

      {/* Download History */}
      {downloadHistory.length > 0 && (
        <div className="bg-white rounded-lg shadow">
          <div className="p-6 border-b border-gray-200">
            <h2 className="text-lg font-semibold text-gray-900">Recent Downloads</h2>
          </div>
          <div className="divide-y divide-gray-200">
            {downloadHistory.slice(-5).reverse().map((item, index) => (
              <div key={index} className="p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="font-medium text-gray-900">{item.name}</p>
                    <p className="text-sm text-gray-500">{item.manufacturer} driver download opened</p>
                  </div>
                  <div className="text-right">
                    <span className="text-sm text-gray-500">{item.timestamp}</span>
                    <a 
                      href={item.url} 
                      target="_blank" 
                      rel="noopener noreferrer"
                      className="block text-xs text-blue-600 hover:text-blue-700 mt-1"
                    >
                      Visit Download Page
                    </a>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}
    </div>
  );
};

export default DriverUpdates;
