import React, { useState, useEffect } from 'react';
import { 
  ComputerDesktopIcon, 
  PrinterIcon, 
  ArrowPathIcon,
  CheckCircleIcon,
  ClockIcon,
  ExclamationTriangleIcon,
  CpuChipIcon,
  ShieldCheckIcon,
  SparklesIcon
} from '@heroicons/react/24/outline';

const Dashboard = ({ driverCount, printerCount, lastDriverCheck, licenseStatus, onRefresh, onNavigate }) => {
  const stats = [
    {
      name: 'Available Drivers',
      value: driverCount,
      icon: ArrowPathIcon,
      color: driverCount > 0 ? 'text-green-600 bg-green-100' : 'text-success-600 bg-success-100',
      description: driverCount > 0 ? 'Driver updates available' : 'All drivers up to date'
    },
    {
      name: 'Connected Printers',
      value: printerCount,
      icon: PrinterIcon,
      color: 'text-primary-600 bg-primary-100',
      description: `${printerCount} printer${printerCount !== 1 ? 's' : ''} detected`
    },
    {
      name: 'System Status',
      value: 'Healthy',
      icon: CheckCircleIcon,
      color: 'text-success-600 bg-success-100',
      description: 'All systems operational'
    },
    {
      name: 'Last Scan',
      value: lastDriverCheck ? new Date(lastDriverCheck).toLocaleTimeString() : 'Never',
      icon: ClockIcon,
      color: 'text-gray-600 bg-gray-100',
      description: 'Last driver scan'
    }
  ];

  const quickActions = [
    {
      name: 'Scan for Drivers',
      description: 'Check for available printer drivers',
      icon: ArrowPathIcon,
      color: 'bg-blue-500 hover:bg-blue-600',
      action: onRefresh
    },
    {
      name: 'Restart Print Spooler',
      description: 'Fix common printer issues',
      icon: PrinterIcon,
      color: 'bg-success-500 hover:bg-success-600',
      action: async () => {
        try {
          await window.electronAPI.restartPrintSpooler();
          await window.electronAPI.writeLog('Print spooler restarted from dashboard');
        } catch (error) {
          console.error('Failed to restart print spooler:', error);
        }
      }
    }
  ];

  return (
    <div className="p-6 animate-fadeInUp">
      {/* Header */}
      <div className="mb-8">
        <div className="flex items-center mb-4">
          <div className="bg-white rounded-lg p-2 mr-4 shadow-md">
            <img 
              src="/assets/greek-squad-logo.jpg" 
              alt="Greek Squad Logo" 
              className="h-10 w-12 object-contain"
            />
          </div>
          <div>
            <h1 className="text-3xl font-bold text-gray-900 mb-2">Dashboard</h1>
            <p className="text-gray-600">Welcome to Greek Driver Updater - Your complete driver management solution</p>
          </div>
        </div>
        <div className="bg-gradient-to-r from-blue-50 to-purple-50 rounded-lg p-4 border border-blue-200">
          <div className="flex items-center">
            <ShieldCheckIcon className="h-5 w-5 text-blue-600 mr-2" />
            <p className="text-blue-800 font-medium">Protected by Greek Squad USA Security</p>
          </div>
        </div>
      </div>

      {/* Stats Cards */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        {stats.map((stat) => {
          const Icon = stat.icon;
          const targetTab = stat.name === 'Available Drivers' ? 'drivers' :
                            stat.name === 'Connected Printers' ? 'printers' :
                            stat.name === 'System Status' ? 'security' : null;
          const clickable = !!targetTab;
          return (
            <div
              key={stat.name}
              className={`bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-lift transition-smooth card-shadow ${clickable ? 'cursor-pointer hover:bg-gray-50' : ''}`}
              role={clickable ? 'button' : undefined}
              tabIndex={clickable ? 0 : undefined}
              onClick={() => clickable && onNavigate && onNavigate(targetTab)}
              onKeyDown={(e) => {
                if (!clickable) return;
                if (e.key === 'Enter' || e.key === ' ') {
                  e.preventDefault();
                  onNavigate && onNavigate(targetTab);
                }
              }}
            >
              <div className="flex items-center">
                <div className={`p-2 rounded-lg ${stat.color}`}>
                  <Icon className="h-6 w-6" />
                </div>
                <div className="ml-4">
                  <p className="text-sm font-medium text-gray-600">{stat.name}</p>
                  <p className="text-2xl font-semibold text-gray-900">{stat.value}</p>
                </div>
              </div>
              <p className="text-sm text-gray-500 mt-2">{stat.description}</p>
            </div>
          );
        })}
      </div>

      {/* Quick Actions */}
      <div className="bg-white rounded-lg shadow-sm border border-gray-200 p-6 hover-lift transition-smooth card-shadow">
        <div className="p-6 border-b border-gray-200">
          <h2 className="text-lg font-semibold text-gray-900">Quick Actions</h2>
          <p className="text-gray-600 mt-1">Common tasks you can perform right now</p>
        </div>
        <div className="p-6">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            {quickActions.map((action) => {
              const Icon = action.icon;
              return (
                <button
                  key={action.name}
                  onClick={action.action}
                  className={`${action.color} text-white p-4 rounded-lg transition-colors flex items-center space-x-3`}
                >
                  <Icon className="h-6 w-6" />
                  <div className="text-left">
                    <p className="font-medium">{action.name}</p>
                    <p className="text-sm opacity-90">{action.description}</p>
                  </div>
                </button>
              );
            })}
          </div>
        </div>
      </div>

      {/* System Information */}
      <div className="bg-white rounded-lg shadow">
        <div className="p-6 border-b border-gray-200">
          <h2 className="text-lg font-semibold text-gray-900 flex items-center">
            <ComputerDesktopIcon className="h-5 w-5 mr-2" />
            System Information
          </h2>
        </div>
        <div className="p-6">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
            <div>
              <h3 className="font-medium text-gray-900 mb-2">Driver Status</h3>
              <div className="space-y-2">
                <div className="flex justify-between">
                  <span className="text-gray-600">Available Drivers:</span>
                  <span className={`font-medium ${driverCount > 0 ? 'text-green-600' : 'text-success-600'}`}>
                    {driverCount}
                  </span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-600">Last Scan:</span>
                  <span className="text-gray-900">{lastDriverCheck || 'Never'}</span>
                </div>
              </div>
            </div>
            <div>
              <h3 className="font-medium text-gray-900 mb-2">Printer Status</h3>
              <div className="space-y-2">
                <div className="flex justify-between">
                  <span className="text-gray-600">Connected Printers:</span>
                  <span className="text-gray-900 font-medium">{printerCount}</span>
                </div>
                <div className="flex justify-between">
                  <span className="text-gray-600">Print Spooler:</span>
                  <span className="text-success-600 font-medium">Running</span>
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </div>
  );
};

export default Dashboard;
