import React, { useState, useEffect } from 'react';
import { 
  ArrowPathIcon, 
  CheckCircleIcon, 
  ExclamationTriangleIcon,
  ArrowDownTrayIcon,
  ClockIcon
} from '@heroicons/react/24/outline';

const AppUpdates = ({ onUpdateComplete }) => {
  const [updates, setUpdates] = useState([]);
  const [loading, setLoading] = useState(false);
  const [updating, setUpdating] = useState(false);
  const [selectedApps, setSelectedApps] = useState(new Set());
  const [lastCheck, setLastCheck] = useState(null);
  const [updateHistory, setUpdateHistory] = useState([]);

  useEffect(() => {
    checkForUpdates();
  }, []);

  const checkForUpdates = async () => {
    setLoading(true);
    try {
      const result = await window.electronAPI.checkUpdates();
      setUpdates(result.updates);
      setLastCheck(new Date().toLocaleString());
      await window.electronAPI.writeLog(`Checked for updates: ${result.updates.length} available`);
    } catch (error) {
      console.error('Failed to check for updates:', error);
      await window.electronAPI.writeLog(`Failed to check for updates: ${error.message}`);
    } finally {
      setLoading(false);
    }
  };

  const upgradeSelected = async () => {
    if (selectedApps.size === 0) return;
    
    setUpdating(true);
    const appsToUpdate = Array.from(selectedApps);
    
    try {
      for (const appId of appsToUpdate) {
        const app = updates.find(u => u.id === appId);
        await window.electronAPI.upgradeApp(appId);
        await window.electronAPI.writeLog(`Updated ${app?.name || appId} to version ${app?.availableVersion}`);
        
        setUpdateHistory(prev => [...prev, {
          name: app?.name || appId,
          version: app?.availableVersion,
          timestamp: new Date().toLocaleString()
        }]);
      }
      
      setSelectedApps(new Set());
      await checkForUpdates();
      if (onUpdateComplete) onUpdateComplete();
    } catch (error) {
      console.error('Failed to upgrade apps:', error);
      await window.electronAPI.writeLog(`Failed to upgrade apps: ${error.message}`);
    } finally {
      setUpdating(false);
    }
  };

  const upgradeAll = async () => {
    setUpdating(true);
    try {
      await window.electronAPI.upgradeApp();
      await window.electronAPI.writeLog(`Updated all applications`);
      
      setUpdateHistory(prev => [...prev, {
        name: 'All Applications',
        version: 'Latest',
        timestamp: new Date().toLocaleString()
      }]);
      
      await checkForUpdates();
      if (onUpdateComplete) onUpdateComplete();
    } catch (error) {
      console.error('Failed to upgrade all apps:', error);
      await window.electronAPI.writeLog(`Failed to upgrade all apps: ${error.message}`);
    } finally {
      setUpdating(false);
    }
  };

  const toggleAppSelection = (appId) => {
    const newSelected = new Set(selectedApps);
    if (newSelected.has(appId)) {
      newSelected.delete(appId);
    } else {
      newSelected.add(appId);
    }
    setSelectedApps(newSelected);
  };

  const selectAll = () => {
    setSelectedApps(new Set(updates.map(u => u.id)));
  };

  const clearSelection = () => {
    setSelectedApps(new Set());
  };

  return (
    <div className="p-6">
      <div className="mb-8">
        <h1 className="text-2xl font-bold text-gray-900">Application Updates</h1>
        <p className="text-gray-600 mt-1">Keep your applications up to date with the latest versions</p>
      </div>

      {/* Action Bar */}
      <div className="bg-white rounded-lg shadow mb-6">
        <div className="p-6 border-b border-gray-200">
          <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between space-y-4 sm:space-y-0">
            <div className="flex items-center space-x-4">
              <button
                onClick={checkForUpdates}
                disabled={loading}
                className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-primary-600 hover:bg-primary-700 disabled:opacity-50"
              >
                <ArrowPathIcon className={`h-4 w-4 mr-2 ${loading ? 'animate-spin' : ''}`} />
                {loading ? 'Checking...' : 'Check for Updates'}
              </button>
              
              {updates.length > 0 && (
                <>
                  <button
                    onClick={upgradeAll}
                    disabled={updating}
                    className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-success-600 hover:bg-success-700 disabled:opacity-50"
                  >
                    <ArrowDownTrayIcon className="h-4 w-4 mr-2" />
                    {updating ? 'Updating...' : 'Update All'}
                  </button>
                  
                  <button
                    onClick={upgradeSelected}
                    disabled={updating || selectedApps.size === 0}
                    className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 disabled:opacity-50"
                  >
                    Update Selected ({selectedApps.size})
                  </button>
                </>
              )}
            </div>
            
            {lastCheck && (
              <div className="flex items-center text-sm text-gray-500">
                <ClockIcon className="h-4 w-4 mr-1" />
                Last check: {lastCheck}
              </div>
            )}
          </div>
        </div>

        {updates.length > 0 && (
          <div className="px-6 py-3 bg-gray-50 border-b border-gray-200">
            <div className="flex items-center justify-between">
              <span className="text-sm text-gray-600">
                {updates.length} update{updates.length !== 1 ? 's' : ''} available
              </span>
              <div className="space-x-2">
                <button
                  onClick={selectAll}
                  className="text-sm text-primary-600 hover:text-primary-700"
                >
                  Select All
                </button>
                <button
                  onClick={clearSelection}
                  className="text-sm text-gray-600 hover:text-gray-700"
                >
                  Clear
                </button>
              </div>
            </div>
          </div>
        )}
      </div>

      {/* Updates List */}
      {updates.length > 0 ? (
        <div className="bg-white rounded-lg shadow">
          <div className="divide-y divide-gray-200">
            {updates.map((update) => (
              <div key={update.id} className="p-6 hover:bg-gray-50">
                <div className="flex items-center justify-between">
                  <div className="flex items-center space-x-4">
                    <input
                      type="checkbox"
                      checked={selectedApps.has(update.id)}
                      onChange={() => toggleAppSelection(update.id)}
                      className="h-4 w-4 text-primary-600 focus:ring-primary-500 border-gray-300 rounded"
                    />
                    <div>
                      <h3 className="text-lg font-medium text-gray-900">{update.name}</h3>
                      <p className="text-sm text-gray-500">ID: {update.id}</p>
                    </div>
                  </div>
                  
                  <div className="text-right">
                    <div className="flex items-center space-x-2">
                      <span className="text-sm text-gray-500">
                        {update.currentVersion} → {update.availableVersion}
                      </span>
                      <ExclamationTriangleIcon className="h-5 w-5 text-warning-500" />
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      ) : (
        <div className="bg-white rounded-lg shadow p-12 text-center">
          <CheckCircleIcon className="h-12 w-12 text-success-500 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">All applications are up to date</h3>
          <p className="text-gray-500">Your system is running the latest versions of all installed applications.</p>
        </div>
      )}

      {/* Update History */}
      {updateHistory.length > 0 && (
        <div className="bg-white rounded-lg shadow mt-6">
          <div className="p-6 border-b border-gray-200">
            <h2 className="text-lg font-semibold text-gray-900">Recent Updates</h2>
          </div>
          <div className="divide-y divide-gray-200">
            {updateHistory.slice(-5).reverse().map((item, index) => (
              <div key={index} className="p-4">
                <div className="flex items-center justify-between">
                  <div>
                    <p className="font-medium text-gray-900">{item.name}</p>
                    <p className="text-sm text-gray-500">Updated to version {item.version}</p>
                  </div>
                  <span className="text-sm text-gray-500">{item.timestamp}</span>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}
    </div>
  );
};

export default AppUpdates;
