import React, { useState, useEffect } from 'react';
import Dashboard from './components/Dashboard';
import AppUpdates from './components/AppUpdates';
import PrinterManager from './components/PrinterManager';
import SecurityCenter from './components/SecurityCenter';
import RealTimeMonitor from './components/RealTimeMonitor';
import Settings from './components/Settings';
import Logs from './components/Logs';
import LoginSignup from './components/LoginSignup';
import LicenseActivation from './components/LicenseActivation';
import AboutUs from './components/AboutUs';
import PrivacyPolicy from './components/PrivacyPolicy';
import TermsOfUse from './components/TermsOfUse';
import SoftwarePolicy from './components/SoftwarePolicy';
import UninstallInstructions from './components/UninstallInstructions';
import { 
  HomeIcon, 
  ArrowDownTrayIcon, 
  PrinterIcon, 
  CogIcon, 
  DocumentTextIcon,
  KeyIcon,
  UserIcon,
  InformationCircleIcon,
  ExclamationTriangleIcon,
  ChevronDownIcon,
  ShieldCheckIcon,
  BellIcon
} from '@heroicons/react/24/outline';

function App() {
  const [activeTab, setActiveTab] = useState('dashboard');
  const [driverCount, setDriverCount] = useState(0);
  const [printerCount, setPrinterCount] = useState(0);
  const [lastDriverCheck, setLastDriverCheck] = useState(null);
  const [licenseStatus, setLicenseStatus] = useState(null);
  const [showLicenseModal, setShowLicenseModal] = useState(false);
  const [currentUser, setCurrentUser] = useState(null);
  const [isAuthenticated, setIsAuthenticated] = useState(false);
  const [authLoading, setAuthLoading] = useState(true);
  const [showHelpMenu, setShowHelpMenu] = useState(false);

  useEffect(() => {
    // Check authentication first
    checkAuthentication();
  }, []);

  useEffect(() => {
    // Load data only if authenticated
    if (isAuthenticated) {
      loadDashboardData();
      checkLicenseStatus();
    }
  }, [isAuthenticated]);

  const loadDashboardData = async () => {
    try {
      // Check for driver updates
      const driverResult = await window.electronAPI.checkDriverUpdates();
      setDriverCount(driverResult.drivers.filter(d => d.hasUpdate).length);
      setLastDriverCheck(new Date().toLocaleString());
      
      // Get printer count
      const printers = await window.electronAPI.getPrinters();
      setPrinterCount(printers.length);
    } catch (error) {
      console.error('Error loading dashboard data:', error);
    }
  };

  const checkAuthentication = async () => {
    try {
      const result = await window.electronAPI.getCurrentUser();
      if (result.success) {
        setCurrentUser(result.user);
        setIsAuthenticated(true);
      } else {
        setIsAuthenticated(false);
      }
    } catch (error) {
      console.error('Error checking authentication:', error);
      setIsAuthenticated(false);
    } finally {
      setAuthLoading(false);
    }
  };

  const checkLicenseStatus = async () => {
    try {
      const status = await window.electronAPI.getLicenseStatus();
      setLicenseStatus(status);
      
      // Show license modal if expired or invalid
      if (status.status === 'expired' || status.status === 'invalid') {
        setShowLicenseModal(true);
      }
    } catch (error) {
      console.error('Error checking license status:', error);
    }
  };

  const handleLicenseActivated = () => {
    setShowLicenseModal(false);
    checkLicenseStatus();
  };

  const handleLoginSuccess = (user) => {
    setCurrentUser(user);
    setIsAuthenticated(true);
  };

  const handleLogout = () => {
    setCurrentUser(null);
    setIsAuthenticated(false);
    setActiveTab('dashboard');
  };

  const handleProfileUpdate = () => {
    checkAuthentication();
  };

  const isFeatureRestricted = () => {
    return licenseStatus && (licenseStatus.status === 'expired' || licenseStatus.status === 'invalid');
  };

  const navigation = [
    { id: 'dashboard', name: 'Dashboard', icon: HomeIcon },
    { id: 'app-updates', name: 'App Updates', icon: ArrowDownTrayIcon },
    { id: 'drivers', name: 'Driver Updates', icon: ArrowDownTrayIcon, restricted: true },
    { id: 'printers', name: 'Printers', icon: PrinterIcon, restricted: true },
    { id: 'security', name: 'Security Center', icon: ShieldCheckIcon },
    { id: 'monitor', name: 'Real-Time Monitor', icon: BellIcon, restricted: true },
    { id: 'profile', name: 'Profile', icon: UserIcon },
    { id: 'license', name: 'License', icon: KeyIcon },
    { id: 'settings', name: 'Settings', icon: CogIcon },
    { id: 'logs', name: 'Logs', icon: DocumentTextIcon },
  ];

  const helpMenuItems = [
    { id: 'about', name: 'About Us', icon: InformationCircleIcon },
    { id: 'privacy', name: 'Privacy Policy', icon: DocumentTextIcon },
    { id: 'terms', name: 'Terms of Use', icon: DocumentTextIcon },
    { id: 'software-policy', name: 'Software Policy', icon: CogIcon },
    { id: 'uninstall', name: 'Uninstall Guide', icon: ExclamationTriangleIcon },
  ];

  const renderContent = () => {
    // Check if feature is restricted and redirect to license page
    if (isFeatureRestricted() && activeTab !== 'dashboard' && activeTab !== 'license' && activeTab !== 'settings') {
      return <LicenseActivation onLicenseActivated={handleLicenseActivated} />;
    }

    switch (activeTab) {
      case 'dashboard':
        return (
          <Dashboard 
            driverCount={driverCount}
            printerCount={printerCount}
            lastDriverCheck={lastDriverCheck}
            licenseStatus={licenseStatus}
            onRefresh={loadDashboardData}
            onNavigate={(tab) => setActiveTab(tab)}
          />
        );
      case 'app-updates':
        return <AppUpdates onUpdateComplete={loadDashboardData} />;
      case 'drivers':
        return isFeatureRestricted() ? 
          <LicenseActivation onLicenseActivated={handleLicenseActivated} /> :
          <DriverUpdates onUpdateComplete={loadDashboardData} />;
      case 'printers':
        return isFeatureRestricted() ? 
          <LicenseActivation onLicenseActivated={handleLicenseActivated} /> :
          <PrinterManager />;
      case 'security':
        return <SecurityCenter />;
      case 'monitor':
        return isFeatureRestricted() ? 
          <LicenseActivation onLicenseActivated={handleLicenseActivated} /> :
          <RealTimeMonitor />;
      case 'profile':
        return <UserProfile currentUser={currentUser} onLogout={handleLogout} onProfileUpdate={handleProfileUpdate} />;
      case 'license':
        return <LicenseActivation onLicenseActivated={handleLicenseActivated} />;
      case 'settings':
        return <Settings licenseStatus={licenseStatus} onLicenseUpdate={checkLicenseStatus} />;
      case 'logs':
        return <Logs />;
      case 'about':
        return <AboutUs />;
      case 'privacy':
        return <PrivacyPolicy />;
      case 'terms':
        return <TermsOfUse />;
      case 'software-policy':
        return <SoftwarePolicy />;
      case 'uninstall':
        return <UninstallInstructions />;
      default:
        return <Dashboard />;
    }
  };

  // Show loading screen while checking authentication
  if (authLoading) {
    return (
      <div className="flex items-center justify-center h-screen bg-gray-50">
        <div className="text-center">
          <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mx-auto mb-4"></div>
          <p className="text-gray-600">Loading Greek Driver Updater...</p>
        </div>
      </div>
    );
  }

  // Show login screen if not authenticated
  if (!isAuthenticated) {
    return <LoginSignup onLoginSuccess={handleLoginSuccess} />;
  }

  return (
    <div className="flex h-screen bg-gray-50">
      {/* Sidebar */}
      <div className="w-64 bg-white shadow-lg">
        <div className="p-6">
          <div className="flex items-center mb-2">
            <img 
              src="/assets/greek-squad-logo.svg" 
              alt="Greek Squad Logo" 
              className="h-8 w-8 mr-2"
            />
            <h1 className="text-xl font-bold text-blue-900">Greek Driver Updater</h1>
          </div>
          <p className="text-sm text-gray-500">by Greek Squad USA</p>
          <p className="text-xs text-blue-600 mt-1">www.greeksquadusa.com</p>
        </div>
        
        <nav className="mt-6">
          {navigation.map((item) => {
            const Icon = item.icon;
            return (
              <button
                key={item.id}
                onClick={() => {
                  if (item.restricted && isFeatureRestricted()) {
                    setActiveTab('license');
                  } else {
                    setActiveTab(item.id);
                  }
                }}
                className={`w-full flex items-center px-6 py-3 text-left transition-colors ${
                  activeTab === item.id
                    ? 'bg-primary-50 text-primary-700 border-r-2 border-primary-500'
                    : item.restricted && isFeatureRestricted()
                    ? 'text-gray-400 hover:bg-gray-50'
                    : 'text-gray-600 hover:bg-gray-50 hover:text-gray-900'
                }`}
              >
                <Icon className="h-5 w-5 mr-3" />
                {item.name}
                {item.restricted && isFeatureRestricted() && (
                  <ExclamationTriangleIcon className="h-4 w-4 ml-auto text-yellow-500" />
                )}
                {item.id === 'drivers' && driverCount > 0 && !isFeatureRestricted() && (
                  <span className="ml-auto bg-danger-500 text-white text-xs rounded-full px-2 py-1">
                    {driverCount}
                  </span>
                )}
                {licenseStatus && item.id === 'license' && (
                  <span className={`ml-auto text-xs px-2 py-1 rounded-full ${
                    licenseStatus.status === 'active' ? 'bg-green-100 text-green-800' :
                    licenseStatus.status === 'trial' ? 'bg-blue-100 text-blue-800' :
                    'bg-red-100 text-red-800'
                  }`}>
                    {licenseStatus.status === 'trial' ? `${licenseStatus.daysRemaining}d` : 
                     licenseStatus.status.charAt(0).toUpperCase() + licenseStatus.status.slice(1)}
                  </span>
                )}
              </button>
            );
          })}
          
          {/* Help & Legal Section */}
          <div className="mt-8 pt-4 border-t border-gray-200">
            <button
              onClick={() => setShowHelpMenu(!showHelpMenu)}
              className="w-full flex items-center justify-between px-6 py-3 text-left text-gray-600 hover:bg-gray-50 hover:text-gray-900 transition-colors"
            >
              <div className="flex items-center">
                <div className="bg-white rounded-lg p-1 mr-3 shadow-md">
                  <img 
                    src="/assets/greek-squad-logo.svg" 
                    alt="Greek Squad Logo" 
                    className="h-8 w-10 object-contain"
                  />
                </div>
                Help & Legal
              </div>
              <ChevronDownIcon className={`h-4 w-4 transition-transform ${
                showHelpMenu ? 'transform rotate-180' : ''
              }`} />
            </button>
            
            {showHelpMenu && (
              <div className="bg-gray-50">
                {helpMenuItems.map((item) => {
                  const Icon = item.icon;
                  return (
                    <button
                      key={item.id}
                      onClick={() => {
                        setActiveTab(item.id);
                        setShowHelpMenu(false);
                      }}
                      className={`w-full flex items-center px-10 py-2 text-left transition-colors ${
                        activeTab === item.id
                          ? 'bg-primary-50 text-primary-700'
                          : 'text-gray-600 hover:bg-gray-100 hover:text-gray-900'
                      }`}
                    >
                      <Icon className="h-4 w-4 mr-3" />
                      {item.name}
                    </button>
                  );
                })}
              </div>
            )}
          </div>
        </nav>
      </div>

      {/* Main content */}
      <div className="flex-1 overflow-hidden">
        <div className="h-full overflow-y-auto">
          {renderContent()}
        </div>
      </div>

      {/* License Modal */}
      {showLicenseModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white rounded-lg shadow-xl max-w-4xl w-full mx-4 max-h-96 overflow-y-auto">
            <div className="p-6 border-b border-gray-200">
              <div className="flex items-center justify-between">
                <h2 className="text-xl font-bold text-gray-900">License Required</h2>
                <button
                  onClick={() => setShowLicenseModal(false)}
                  className="text-gray-400 hover:text-gray-600"
                >
                  ×
                </button>
              </div>
            </div>
            <div className="p-6">
              <LicenseActivation onLicenseActivated={handleLicenseActivated} />
            </div>
          </div>
        </div>
      )}
    </div>
  );
}

export default App;
